<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
	version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<!--
 ==============================================================================
 == Simple Text Race Results Report Generation
 ==============================================================================
 -->

<xsl:import href="scoringutils.xsl"/>

<xsl:output method="text"/>


<!--
 ==============================================================================
 == client parameters and template processing options
 ==============================================================================
 -->

<!-- default standings organization is by class -->
<xsl:param name="organizationType" select="'ByClass'"/>

<!-- optional racer name annotations - specify yes to enable -->
<xsl:param name="includeRacerClass" select="'no'"/>
<xsl:param name="includeRacerAffiliation" select="'no'"/>

<xsl:param name="includeRacePoints" select="'yes'"/>

<!-- debug/trace options -->
<xsl:param name="DEBUG" select="'no'"/>


<!--
 ==============================================================================
 == global variables from ACES_RaceScoring
 ==============================================================================
 -->

<xsl:variable name="scoringRoot" select="/ACES_RaceScoring"/>

<xsl:variable name="theRace" select="$scoringRoot/race[1]"/>

<xsl:variable name="reportTitle">
	<xsl:call-template name="raceDescription">
		<xsl:with-param name="race" select="$scoringRoot/race[1]"/>
	</xsl:call-template>
</xsl:variable>

<xsl:variable name="standingsDescription">
	<xsl:text>Standings </xsl:text>
	<xsl:value-of select="$organizationDescription"/>
</xsl:variable>

<xsl:variable name="allCompetitors" 
	select="$scoringRoot/competitors/competitor"/>

<xsl:variable name="standings" 
	select="$scoringRoot/standings[@name=$organizationType]"/>
<xsl:variable name="organizationDescription" select="$standings/description"/>

<xsl:variable name="nRuns" select="$theRace/nRuns"/>


<!--
  =============================================================================
 == root template: ACES_RaceScoring -> race standings report
 ==============================================================================
 -->

<xsl:template match="/">

	<!-- correctness checking (?overzealous?!) - this template isn't really strictly necessary -->
	<xsl:if test="count(ACES_RaceScoring) != 1">
		<xsl:message terminate="yes">****FATAL ERROR: cannot process &lt;<xsl:value-of select="name(child::*[1])"/>&gt; - &lt;ACES_RACEScoring&gt; scoring data required</xsl:message>
	</xsl:if>
	<xsl:if test="count($standings) != 1">
		<xsl:message terminate="yes">***FATAL ERROR: no <xsl:value-of select="$organizationType"/> &lt;standings&gt; available for race <xsl:value-of select="$theRace/raceID"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>

	<xsl:apply-templates select="ACES_RaceScoring"/>

</xsl:template> 


<!--
 ==============================================================================
 == ACES_RaceScoring -> race standings report
 ==============================================================================
 -->

<xsl:template match="ACES_RaceScoring">

<xsl:call-template name="generatePageHeader"/>

<xsl:call-template name="generateReportHeader"/>
<xsl:call-template name="generateReportBody"/>
<xsl:call-template name="generateReportFooter"/>

<xsl:call-template name="generatePageFooter"/>

</xsl:template>

<xsl:template name="generatePageHeader">
	<xsl:value-of select="$reportTitle"/>
	<xsl:text>&#xA;</xsl:text>
</xsl:template>

<xsl:template name="generatePageFooter">
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Produced by </xsl:text>
	<xsl:value-of select="$scoringRoot/auxInfo/softwareDescription"/>
	<xsl:text>&#xA;</xsl:text>
</xsl:template>


<!--
 ==============================================================================
 == Report header section
 ==============================================================================
 -->

<xsl:template name="generateReportHeader">

	<xsl:variable name="competitorSummary" 
		select="$standings/standingsSummary/competitorSummary"/>

	<xsl:value-of select="$competitorSummary[@category='all']/@nStarters"/><xsl:text> competitors (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> women)</xsl:text>
	<xsl:text>&#xA;</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='all']/@nQualifiers"/><xsl:text> finishers (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nQualifiers"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nQualifiers"/><xsl:text> women)</xsl:text>
	<xsl:text>&#xA;</xsl:text>

</xsl:template>


<!--
 ==============================================================================
 == Report footer section
 ==============================================================================
 -->

<xsl:template name="generateReportFooter">

	<xsl:variable name="classes">
		<xsl:call-template name="classesDescription">
			<xsl:with-param name="classes" select="race/classes"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:text>&#xA;</xsl:text>

	<xsl:text>Scoring Details:</xsl:text>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Race points: F=</xsl:text><xsl:value-of select="scoring/fValue"/>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Classes: </xsl:text><xsl:value-of select="$classes"/>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Courses: </xsl:text><xsl:value-of select="race/courses"/>
	<xsl:text>&#xA;</xsl:text>

</xsl:template>


<!--
 ==============================================================================
 == Report body
 ==============================================================================
 -->

<xsl:template name="generateReportBody">

	<xsl:if test="$DEBUG='yes'">
		<xsl:message>***PROCESSING &lt;standings&gt; organization <xsl:value-of select="$organizationType"/></xsl:message>
	</xsl:if>

	<xsl:text>&#xA;</xsl:text>

	<xsl:value-of select="$standingsDescription"/>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Format: n. name - time(s)</xsl:text>
	<xsl:if test="$includeRacePoints='yes'">
		<xsl:text> - race points</xsl:text>
	</xsl:if>
	<xsl:text>&#xA;</xsl:text>

	<xsl:apply-templates select="$standings"/>

</xsl:template>

<xsl:template match="standings">

	<xsl:variable name="nGroups" select="count(groupStandings)"/>
	<xsl:variable name="includeCategoryHeaders" select="not (@singleCategory='yes')"/>  <!-- and nGroups > 1 -->

	<xsl:for-each select="groupStandings">
		<xsl:call-template name="processResultGroup">
			<xsl:with-param name="includeCategoryHeaders" select="$includeCategoryHeaders"/>
		</xsl:call-template>
	</xsl:for-each>

</xsl:template>

<xsl:template name="processResultGroup">
<xsl:param name="includeCategoryHeaders"/>
	<xsl:if test="$includeCategoryHeaders">
		<xsl:text>&#xA;</xsl:text>
		<xsl:value-of select="description"/>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>
	<xsl:for-each select="racerResult">
		<xsl:call-template name="processRacerResult"/>
	</xsl:for-each>
	<!-- for race results, we don't care about optional category summary - rare to have ties in the group -->
</xsl:template>

<xsl:template name="processRacerResult">

	<xsl:variable name="racerID" select="@racerID"/>
	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>

	<xsl:variable name="status" select="$racer/@status"/>
	<xsl:variable name="racerName">
		<xsl:call-template name="composeRacerNameEntry">
			<xsl:with-param name="racer" select="$racer"/>
			<xsl:with-param name="ageClass" select="@ageClass"/>  <!-- optional attr in OP/SS class standings -->
			<xsl:with-param name="includeRacerClass" 
				select="$includeRacerClass='yes'"/>
			<xsl:with-param name="includeRacerAffiliation" 
				select="$includeRacerAffiliation='yes'"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:if test="$status!='DNS'">
	<xsl:choose>
		<xsl:when test="$status='QLF'">
			<xsl:call-template name="positionString"/><xsl:text>. </xsl:text>
		</xsl:when>
		<xsl:otherwise>
			<td>- </td>
		</xsl:otherwise>
	</xsl:choose>
	<xsl:value-of select="$racerName"/>
	<xsl:text> - </xsl:text>
	<xsl:value-of select="$racer/result"/>
	<xsl:if test="$nRuns=2">
		<xsl:text> (</xsl:text>
		<xsl:value-of select="$racer/run1"/><xsl:text>, </xsl:text>
		<xsl:value-of select="$racer/run2"/>
		<xsl:text>)</xsl:text>
	</xsl:if>

	<xsl:if test="$includeRacePoints='yes' and $status='QLF'">
		<xsl:text> - </xsl:text>
		<xsl:value-of select="racePoints"/>
	</xsl:if>
	<xsl:text>&#xA;</xsl:text>
	</xsl:if>

</xsl:template>

</xsl:stylesheet>
