<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
	version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<!--
 ==============================================================================
 == Simple Text Series Standings Report Generation
 ==============================================================================
 -->

<xsl:import href="scoringutils.xsl"/>

<xsl:output method="text"/>


<!--
 ==============================================================================
 == client parameters and template processing options
 ==============================================================================
 -->

<!-- default standings organization is by class -->
<xsl:param name="organizationType" select="'ByClass'"/>

<!-- optional racer name annotations - specify yes to enable -->
<xsl:param name="includeRacerAffiliation" select="'no'"/>
<xsl:param name="includeRacerClass" select="'no'"/>

<xsl:param name="includeRacePoints" select="'yes'"/>

<!-- debug/trace options -->
<xsl:param name="DEBUG" select="'no'"/>


<!--
 ==============================================================================
 == global variables from ACES_SeriesScoring
 ==============================================================================
 -->

<xsl:variable name="scoringRoot" select="/ACES_SeriesScoring"/>

<xsl:variable name="theEvent" select="$scoringRoot/event[1]"/>

<xsl:variable name="reportTitle">
	<xsl:call-template name="eventDescription">
		<xsl:with-param name="event" select="$theEvent"/>
	</xsl:call-template>
</xsl:variable>

<xsl:variable name="standingsDescription">
	<xsl:value-of select="$theEvent/standingsType"/>
	<xsl:text> Standings </xsl:text>
	<xsl:value-of select="$organizationDescription"/>
</xsl:variable>

<xsl:variable name="allCompetitors" 
	select="$scoringRoot/competitors/competitor"/>

<xsl:variable name="standings" 
	select="$scoringRoot/standings[@name=$organizationType]"/>
<xsl:variable name="organizationDescription" select="$standings/description"/>

<xsl:variable name="seriesRaces" select="$theEvent/races/race"/>
<xsl:variable name="nRaces" select="count($seriesRaces)"/>


<!--
 ==============================================================================
 == root template: ACES_SeriesScoring-> series standings report
 ==============================================================================
 -->

<!-- note that this isn't actually necessary -->
<xsl:template match="/">

	<!-- correctness checking (?overzealous?!) - this template isn't really strictly necessary -->
	<xsl:if test="count(ACES_SeriesScoring) != 1">
		<xsl:message terminate="yes">****FATAL ERROR: cannot process &lt;<xsl:value-of select="name(child::*[1])"/>&gt; - &lt;ACES_SeriesScoring&gt; scoring data required</xsl:message>
	</xsl:if>
	<xsl:if test="count($standings) != 1">
		<xsl:message terminate="yes">***FATAL ERROR: no <xsl:value-of select="$organizationType"/> &lt;standings&gt; available for event<xsl:value-of select="$theEvent/name"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>

	<xsl:apply-templates select="ACES_SeriesScoring"/>

</xsl:template>


<!--
 ==============================================================================
 == ACES_SeriesScoring -> series/event standings report
 ==============================================================================
 -->

<xsl:template match="ACES_SeriesScoring">

<xsl:call-template name="generatePageHeader"/>

<xsl:call-template name="generateReportHeader"/>
<xsl:call-template name="generateReportBody"/>
<xsl:call-template name="generateReportFooter"/>

<xsl:call-template name="generatePageFooter"/>

</xsl:template>


<xsl:template name="generatePageHeader">
	<xsl:value-of select="$reportTitle"/>
	<xsl:text>&#xA;</xsl:text>
</xsl:template>

<xsl:template name="generatePageFooter">
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Produced by </xsl:text>
	<xsl:value-of select="$scoringRoot/auxInfo/softwareDescription"/>
</xsl:template>


<!--
 ==============================================================================
 == Report header section
 ==============================================================================
 -->

<xsl:template name="generateReportHeader">

	<xsl:variable name="competitorSummary" 
		select="$standings/standingsSummary/competitorSummary"/>
	<xsl:variable name="competitorStarts"
		select="$standings/standingsSummary/competitorStarts"/>
	<xsl:variable name="raceStarts"
		select="$standings/standingsSummary/raceStarts"/>	

	<xsl:text>&#xA;</xsl:text>

	<xsl:value-of select="$theEvent/standingsType"/><xsl:text> Standings Summary</xsl:text>
	<xsl:text>&#xA;</xsl:text>

	<xsl:choose>
		<xsl:when test="$nRaces=1">
			<xsl:text>Single-race event scoring.</xsl:text>
		</xsl:when>
		<xsl:otherwise>
			<xsl:text></xsl:text><xsl:value-of select="$nRaces"/><xsl:text> races scored.</xsl:text>
		</xsl:otherwise>
	</xsl:choose>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Scoring policy: </xsl:text>
	<!-- DROP: <xsl:value-of select="$organizationType"/><xsl:text> - </xsl:text> -->
	<xsl:value-of select="scoring/scoringPolicy"/><xsl:text>.</xsl:text>
	<xsl:text>&#xA;</xsl:text>

	<!-- Season scoring: total number of starts, ave starts per racer, starts per discipline -->
	<xsl:if test="$raceStarts">
		<xsl:text>&#xA;</xsl:text>
		<!-- assert( "$theEvent/standingsType='Season'" ) -->
		<xsl:variable name="allRaceStarts" select="$raceStarts[@category='all']"/>
		<xsl:text>Total number of starts: </xsl:text><xsl:value-of select="$allRaceStarts/@total"/>
		<xsl:text>&#xA;</xsl:text>
		<xsl:text>Average starters per race: </xsl:text><xsl:value-of select="$allRaceStarts/@ave"/>
		<xsl:text>&#xA;</xsl:text>
		<xsl:for-each select="$raceStarts[@category != 'all']">
			<xsl:text>Ave starters </xsl:text><xsl:value-of select="@category"/><xsl:text>: </xsl:text>
			<xsl:value-of select="@ave"/>
			<xsl:text> (</xsl:text><xsl:value-of select="@total"/><xsl:text> </xsl:text>
			<xsl:value-of select="@category"/><xsl:text> starts, </xsl:text>
			<xsl:value-of select="@nRaces"/><xsl:text> races)</xsl:text>
			<xsl:text>&#xA;</xsl:text>
		</xsl:for-each>
	</xsl:if>

	<xsl:text>&#xA;</xsl:text>

	<xsl:text>Total number of competitors: </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='all']/@nStarters"/><xsl:text> (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> women)</xsl:text>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Number of qualifiers: </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='all']/@nQualifiers"/><xsl:text> (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nQualifiers"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nQualifiers"/><xsl:text> women)</xsl:text>
	<xsl:if test="$theEvent/standingsType='Season'">
		<xsl:text>&#xA;</xsl:text>
<!-- ### VER 1 WAS:
		<xsl:text>Average number of starts per racer: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='all']/@ave"/><xsl:text> (</xsl:text>
		<xsl:value-of select="$competitorStarts[@category='M']/@ave"/><xsl:text> men, </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='F']/@ave"/><xsl:text> women)</xsl:text>
-->
		<xsl:variable name="allRacerStarts" select="$competitorStarts[@category='all']"/>
		<xsl:text>Average starts per racer: </xsl:text><xsl:value-of select="$allRacerStarts/@ave"/>
		<xsl:text>&#xA;</xsl:text>
		<xsl:text>Ave starts men: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='M']/@ave"/>
		<xsl:text> (</xsl:text><xsl:value-of select="$competitorStarts[@category='M']/@total"/>
		<xsl:text> starts, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> competitors)</xsl:text>
		<xsl:text>&#xA;</xsl:text>
		<xsl:text>Ave starts women: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='F']/@ave"/>
		<xsl:text> (</xsl:text><xsl:value-of select="$competitorStarts[@category='F']/@total"/>
		<xsl:text> starts, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> competitors)</xsl:text>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>
	<xsl:text>&#xA;</xsl:text>

	<!-- DIVISIONS CUP SUMMARY (special-case hack, oh well) -->
	<xsl:variable name="teamResults" 
		select="$standings/standingsSummary/teamSummary"/>
	<xsl:if test="$teamResults">
		<xsl:text>&#xA;</xsl:text>
		<!-- assert( "$organizationType='ByAffiliation' and policy is Divisions Cup..." ) -->
		<xsl:text>Divisions Cup Standings (points per start):</xsl:text>
		<xsl:text>&#xA;</xsl:text>
		<xsl:for-each select="$teamResults">
			<xsl:choose>
				<xsl:when test="@status='QLF'">
					<xsl:value-of select="@posn"/><xsl:text>. </xsl:text>
				</xsl:when>
				<xsl:otherwise>
					<xsl:text>- </xsl:text>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:value-of select="description"/>  <!-- @name has short name -->
			<xsl:text> - </xsl:text>
			<xsl:value-of select="@ave"/>  <!-- ave: [points per start = total points / numStarts -->
			<xsl:text> (</xsl:text>
			<xsl:value-of select="@nRacers"/><xsl:text> racers, </xsl:text>
			<xsl:value-of select="@nStarts"/><xsl:text> starts, </xsl:text>
			<xsl:value-of select="@total"/><xsl:text> points)</xsl:text>
			<xsl:text>&#xA;</xsl:text>
		</xsl:for-each>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>

	<!-- MEDAL COUNTS SUMMARY (special-case hack, oh well) -->
	<xsl:variable name="teamMedals" 
		select="$standings/standingsSummary/teamMedals"/>
	<xsl:if test="$teamMedals">
		<xsl:text>&#xA;</xsl:text>
		<!-- assert( "$organizationType='ByAffiliation' and policy is Medal Counts..." ) -->
		<xsl:text>Medal Counts:</xsl:text>
		<xsl:text>&#xA;</xsl:text>
		<xsl:for-each select="$teamMedals">
			<xsl:choose>
				<xsl:when test="@status='QLF'">
					<xsl:value-of select="@posn"/><xsl:text>. </xsl:text>
				</xsl:when>
				<xsl:otherwise>
					<xsl:text>- </xsl:text>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:value-of select="description"/>  <!-- @name has short name -->
			<xsl:text> - </xsl:text>
			<xsl:choose>
			  <xsl:when test="@total">
			    <xsl:value-of select="@total"/>  <!-- medals -->
			  </xsl:when>
				<xsl:otherwise>
					<xsl:text>(no medals)</xsl:text>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:text> - </xsl:text>
            <xsl:if test="@gold">
			  <xsl:value-of select="@gold"/><xsl:text> gold</xsl:text>
            </xsl:if>
            <xsl:if test="@silver">
              <xsl:if test="@gold"><xsl:text>, </xsl:text></xsl:if>
			  <xsl:value-of select="@silver"/><xsl:text> silver</xsl:text>
            </xsl:if>
            <xsl:if test="@bronze">
              <xsl:if test="@gold or @silver"><xsl:text>, </xsl:text></xsl:if>
			  <xsl:value-of select="@bronze"/><xsl:text> bronze</xsl:text>
            </xsl:if>
			<xsl:text> (</xsl:text>
			<xsl:value-of select="@nRacers"/><xsl:text> racers</xsl:text>
			<xsl:if test="@nMedalists">
			  <xsl:text>, </xsl:text>
			  <xsl:value-of select="@nMedalists"/><xsl:text> medalists</xsl:text>
			</xsl:if>
			<xsl:text>)</xsl:text>
			<xsl:text>&#xA;</xsl:text>
		</xsl:for-each>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>

</xsl:template>


<!--
 ==============================================================================
 == Report footer section
 ==============================================================================
 -->

<xsl:template name="generateReportFooter">

	<xsl:variable name="classes">
		<xsl:call-template name="classesDescription">
			<xsl:with-param name="classes" select="event/classes"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:variable name="scoringElement">
		<xsl:call-template name="scoringElementDescription">
			<xsl:with-param name="scoringElementName" select="scoring/scoringElement"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:text>&#xA;</xsl:text>

	<xsl:text>Scoring Details:</xsl:text>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Classes: </xsl:text><xsl:value-of select="$classes"/>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Scoring element: </xsl:text><xsl:value-of select="$scoringElement"/>
	<xsl:text>&#xA;</xsl:text>
	<xsl:text>Scoring policy: </xsl:text><xsl:value-of select="scoring/scoringPolicy"/>
	<xsl:text>&#xA;</xsl:text>

	<xsl:text>&#xA;</xsl:text>

	<xsl:text>Scored Races</xsl:text>
	<xsl:text>&#xA;</xsl:text>
	<xsl:for-each select="$seriesRaces">
		<xsl:value-of select="position()"/>
		<xsl:text>. </xsl:text>
		<xsl:call-template name="raceDescription">
			<xsl:with-param name="race" select="."/>
		</xsl:call-template>
		<xsl:text> - </xsl:text>
		<xsl:value-of select="courses"/>
		<xsl:text>&#xA;</xsl:text>
	</xsl:for-each>

	<xsl:variable name="multiCategoryRacers" 
		select="$standings/groupStandings/racerResult[@multipleResults='yes']"/>
			<!-- WAS: $allCompetitors[@nStarts] -->
	<xsl:if test="$multiCategoryRacers">
		<xsl:text>&#xA;</xsl:text>
		<xsl:choose>
			<xsl:when test="$organizationType='ByClass'">
				<xsl:text>* Racers with starts in both Open seed and age class:</xsl:text>
				<xsl:text>&#xA;</xsl:text>
				<xsl:text>Racer name - total number of starts (total number of finishes)</xsl:text>
				<xsl:text>&#xA;</xsl:text>
<!-- ####### TO DO [DJL 16-Nov-2002] ########### -->
<xsl:text>***NOTE: Need to fix sorting!!</xsl:text>
<xsl:text>&#xA;</xsl:text>
				<xsl:call-template name="generateMultiClassRacersReport">
					<xsl:with-param name="multiClassRacers" select="$multiCategoryRacers"/>
					<xsl:with-param name="index" select="1"/>
					<xsl:with-param name="prevAgeClass" select="'NONE'"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<!-- //probably overzealous; if this is really serious, should detect long before now, eh??
				<xsl:message terminate="yes">***ERROR: should not have racers in multiple categories!!</xsl:message>
				-->
				<xsl:text>***WARNING:</xsl:text><xsl:value-of select="count($multiCategoryRacers)"/><xsl:text> racers have starts in multiple categories - should not be!!!</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:if>

</xsl:template>

<xsl:template name="generateMultiClassRacersReport">
<xsl:param name="multiClassRacers"/>
<xsl:param name="index"/>
<xsl:param name="prevAgeClass"/>

	<xsl:variable name="racerResult" select="$multiClassRacers[$index]"/>
	<xsl:variable name="racerID" select="$racerResult/@racerID"/>
	<xsl:variable name="currentClass" select="$racerResult/../@name"/>  <!-- groupStandings category name attr -->

	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>
	<xsl:variable name="racerAgeClass" select="$racer/class"/>

	<!-- report on age clases; skip ability class results, they're the dup guys -->
	<xsl:if test="$racerAgeClass=$currentClass">

		<!-- emit separator and category header between age classes -->
		<xsl:if test="$racerAgeClass != $prevAgeClass">
			<!-- ISSUE: Just have name of the class at this point; we've lost its (category) description... -->
			<xsl:text>&#xA;</xsl:text>
			<xsl:value-of select="$racerAgeClass"/>
			<xsl:text>&#xA;</xsl:text>
		</xsl:if>

		<xsl:value-of select="$racer/lastName"/>, <xsl:value-of select="$racer/firstName"/>
		<xsl:text> - </xsl:text>
		<xsl:value-of select="$racer/@nStarts"/>
		<xsl:text> (</xsl:text>
		<xsl:value-of select="$racer/@nFinishes"/>
		<xsl:text>)</xsl:text>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>

	<xsl:if test="$index != count($multiClassRacers)">
		<xsl:call-template name="generateMultiClassRacersReport">
			<xsl:with-param name="multiClassRacers" select="$multiClassRacers"/>
			<xsl:with-param name="index" select="$index + 1"/>
			<xsl:with-param name="prevAgeClass" select="$currentClass"/>
		</xsl:call-template>
	</xsl:if>

</xsl:template>


<!--
 ==============================================================================
 == Report body
 ==============================================================================
 -->

<xsl:template name="generateReportBody">

	<xsl:if test="$DEBUG='yes'">
		<xsl:message>***PROCESSING &lt;standings&gt; organization <xsl:value-of select="$organizationType"/></xsl:message>
	</xsl:if>

	<xsl:text>&#xA;</xsl:text>
	<xsl:value-of select="$standingsDescription"/>
	<xsl:text>&#xA;</xsl:text>

	<xsl:text>Format: n. name - </xsl:text>
	<xsl:call-template name="scoringElementLabel">
		<xsl:with-param name="scoringElementName" select="scoring/scoringElement"/>
	</xsl:call-template>
	<xsl:text> - St(Fn) - race results</xsl:text>
	<xsl:text>&#xA;</xsl:text>
	<xsl:call-template name="generateRacesList"/>

	<xsl:text>&#xA;</xsl:text>
	<xsl:apply-templates select="$standings"/>

</xsl:template>

<xsl:template name="generateRacesList">
	<xsl:text>   Races: </xsl:text>
	<xsl:for-each select="$seriesRaces">
		<xsl:if test="position() != 1">
			<xsl:text>, </xsl:text>
		</xsl:if>
		<xsl:value-of select="location/@name"/>
		<xsl:text> </xsl:text><xsl:value-of select="discipline"/>
	</xsl:for-each>
</xsl:template>

<xsl:template match="standings">

	<xsl:variable name="nGroups" select="count(groupStandings)"/>
	<xsl:variable name="includeCategoryHeaders" select="not (@singleCategory='yes')"/>  <!-- and nGroups > 1 -->

	<xsl:variable name="nColumns" select="5 + $nRaces"/>

	<xsl:for-each select="groupStandings">
		<xsl:call-template name="processResultGroup">
			<xsl:with-param name="includeCategoryHeaders" select="$includeCategoryHeaders"/>
			<xsl:with-param name="nColumns" select="$nColumns"/>
		</xsl:call-template>
	</xsl:for-each>

</xsl:template>

<xsl:template name="processResultGroup">
<xsl:param name="includeCategoryHeaders"/>
<xsl:param name="nColumns"/>

	<xsl:if test="$includeCategoryHeaders">
		<xsl:text>&#xA;</xsl:text>
		<xsl:value-of select="description"/>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>

	<xsl:for-each select="racerResult">
		<xsl:call-template name="processRacerResult"/>
	</xsl:for-each>

	<!-- optional category summary -->
	<xsl:variable name="nQualifiers" select="@nQualifiers"/>
	<xsl:if test="$nQualifiers">
		<xsl:variable name="nCategoryCompetitors" select="count(racerResult)"/>
		<xsl:text>(</xsl:text>
		<xsl:value-of select="$nQualifiers"/>
		<xsl:if test="$nQualifiers != $nCategoryCompetitors">
			<xsl:text> of </xsl:text>
			<xsl:value-of select="$nCategoryCompetitors"/>
		</xsl:if>
		<xsl:text> competitors</xsl:text>
		<xsl:if test="$nQualifiers != $nCategoryCompetitors">
			<xsl:text> qualified</xsl:text>
		</xsl:if>
		<xsl:text>)</xsl:text>
		<xsl:text>&#xA;</xsl:text>
	</xsl:if>

</xsl:template>

<xsl:template name="processRacerResult">

	<xsl:variable name="racerID" select="@racerID"/>
	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>

	<xsl:variable name="status" select="@status"/>
	<xsl:variable name="racerName">
		<xsl:call-template name="composeRacerNameEntry">
			<xsl:with-param name="racer" select="$racer"/>
			<xsl:with-param name="ageClass" select="@ageClass"/>  <!-- optional attr in OP/SS class standings -->
			<xsl:with-param name="includeRacerClass" 
				select="$includeRacerClass='yes'"/>
			<xsl:with-param name="includeRacerAffiliation" 
				select="$includeRacerAffiliation='yes'"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:if test="$status!='DNS'">
	<xsl:choose>
		<xsl:when test="$status='QLF'">
			<xsl:call-template name="positionString"/><xsl:text>. </xsl:text>
			<xsl:value-of select="$racerName"/>
			<xsl:text> - </xsl:text>
			<xsl:call-template name="composeRacerScoreEntry">
				<xsl:with-param name="score" select="score"/>
				<xsl:with-param name="valueIfZero" select="'0'"/>
			</xsl:call-template>
		</xsl:when>
		<xsl:otherwise>
			<xsl:text>- </xsl:text>
			<xsl:value-of select="$racerName"/>
			<xsl:text> - </xsl:text>
			<!-- ISSUE: report score if present, even though not qualified?? -->
			<xsl:text>DNQ</xsl:text>   <!-- (empty score ) -->
		</xsl:otherwise>
	</xsl:choose>
	<xsl:text> - </xsl:text>
	<xsl:call-template name="composeSeriesStartFinishEntry">
		<xsl:with-param name="racerResult" select="."/>
		<xsl:with-param name="racer" select="$racer"/>
	</xsl:call-template>
	<xsl:text> - </xsl:text>
	<xsl:call-template name="generateRaceScoreEntries">
		<xsl:with-param name="racerResult" select="."/>
	</xsl:call-template>
	<xsl:text>&#xA;</xsl:text>
	</xsl:if>

</xsl:template>

<xsl:template name="generateRaceScoreEntries">
<xsl:param name="racerResult"/>
<xsl:param name="index" select="1"/>

	<xsl:variable name="raceScore" select="$racerResult/rsScore[@ri=$index]"/>

	<xsl:if test="$index != 1">
		<xsl:text>, </xsl:text>
	</xsl:if>

	<xsl:choose>
		<xsl:when test="$raceScore">
			<!-- #### TO DO (someday): if $racerResult/@counted='no' -> flag excluded score ####-->
			<xsl:value-of select="$raceScore"/>
			<xsl:if test="$raceScore/@posn">
				<xsl:text>(</xsl:text>
				<xsl:call-template name="seriesStartPositionString">
					<xsl:with-param name="posnAttr" select="$raceScore/@posn"/>
				</xsl:call-template>
				<xsl:text>)</xsl:text>
			</xsl:if>
		</xsl:when>
		<xsl:otherwise>
			<xsl:text>x</xsl:text>  <!-- used to use a "-" for this... -->
		</xsl:otherwise>
	</xsl:choose>

	<xsl:if test="not ($index = $nRaces)">
		<xsl:call-template name="generateRaceScoreEntries">
			<xsl:with-param name="racerResult" select="$racerResult"/>
			<xsl:with-param name="index" select="$index + 1"/>
		</xsl:call-template>
	</xsl:if>

</xsl:template>

</xsl:stylesheet>

