<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
	version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<!-- General-purpose scoring report utilities -->

<!--
 ==============================================================================
 == race information utilities
 ==============================================================================
-->

<!-- Answer a description of the specified race -->
<xsl:template name="raceDescription">
<xsl:param name="race"/>
	<!-- assert count($race) = 1 -->
	<xsl:value-of select="$race/location"/>
	<xsl:text> </xsl:text>
	<xsl:value-of select="$race/discipline"/>
	<xsl:text> - </xsl:text>
	<xsl:value-of select="$race/date"/>
</xsl:template>

<!-- Answer a description of the race class system -->
<xsl:template name="classesDescription">
<xsl:param name="classes"/>
	<xsl:variable name="classesLiteral" select="$classes/@name"/>
	<xsl:choose>
	<!-- optionally activate clauses to override built-in descriptions:
		<xsl:when test="$classesLiteral='NationalClasses'">
			<xsl:text>USSA masters 5-year age classes (20-85+)</xsl:text>
		</xsl:when>
		<xsl:when test="$classesLiteral='InternationalClasses'">
			<xsl:text>International masters 5-year age classes (30-85+)</xsl:text>
		</xsl:when>
		<xsl:when test="$classesLiteral='FarWestClasses'">
			<xsl:text>Far West masters 10-year age classes</xsl:text>
		</xsl:when>
		<xsl:when test="$classesLiteral='IntermountainClasses'">
			<xsl:text>Intermountain masters 5-year age classes</xsl:text>
		</xsl:when>
		<xsl:when test="$classesLiteral='RockyMountainClasses'">
			<xsl:text>Rocky Mountain masters 5-year age classes</xsl:text>
		</xsl:when>
	-->
		<xsl:when test="false()"/>  <!-- NOP placeholder for XSLT syntax -->
		<xsl:otherwise>
			<xsl:value-of select="$classes"/>
		</xsl:otherwise>
	</xsl:choose>
</xsl:template>


<!--
 ==============================================================================
 == combined and series event information utilities
 ==============================================================================
-->

<!-- Answer a description of the specified event -->
<xsl:template name="eventDescription">
<xsl:param name="event"/>
	<!-- assert count($event) = 1 -->
	<xsl:value-of select="$event/name"/>
</xsl:template>


<!--
 ==============================================================================
 == scoring information utilities
 ==============================================================================
-->

<!-- Answer a description of the scoring element -->
<xsl:template name="scoringElementDescription">
<xsl:param name="scoringElementName"/>
	<xsl:choose>
		<xsl:when test="$scoringElementName='Time'">
			<xsl:text>Total time</xsl:text>
		</xsl:when>
		<xsl:when test="$scoringElementName='AdjustedTime'">
			<xsl:text>Adjusted total time</xsl:text>
		</xsl:when>
		<xsl:when test="$scoringElementName='Points'">
			<xsl:text>World Cup points</xsl:text>
		</xsl:when>
		<xsl:when test="$scoringElementName='RacePoints'">
			<xsl:text>Race points</xsl:text>
		</xsl:when>
		<xsl:when test="$scoringElementName='DivisionsCupPoints'">
			<xsl:text>Division's Cup points</xsl:text>
		</xsl:when>
		<xsl:otherwise>
			<xsl:value-of select="$scoringElementName"/>
		</xsl:otherwise>
	</xsl:choose>
</xsl:template>

<!-- Answer the column label for the scoring element -->
<xsl:template name="scoringElementLabel">
<xsl:param name="scoringElementName"/>
	<xsl:choose>
		<xsl:when test="$scoringElementName='Time' or $scoringElementName='AdjustedTime'">
			<xsl:text>Time</xsl:text>
		</xsl:when>
		<xsl:when test="$scoringElementName='Points' or $scoringElementName='RacePoints'">
			<xsl:text>Points</xsl:text>
		</xsl:when>
		<xsl:when test="$scoringElementName='DivisionsCupPoints'">
			<xsl:text>Points</xsl:text>
		</xsl:when>
		<xsl:otherwise>
			<xsl:value-of select="$scoringElementName"/>
		</xsl:otherwise>
	</xsl:choose>
</xsl:template>


<!--
 ==============================================================================
 == racer and racerResult utilities
 ==============================================================================
-->

<xsl:template name="composeRacerNameEntry">
<xsl:param name="racer"/>
<xsl:param name="ageClass" select='""'/>
<xsl:param name="includeRacerClass" select="false()"/>
<xsl:param name="includeRacerAffiliation" select="false()"/>

	<xsl:value-of select="$racer/lastName"/>
	<xsl:text>, </xsl:text>
	<xsl:value-of select="$racer/firstName"/>

	<xsl:variable name="classAnnotation">
		<xsl:choose>
			<xsl:when test='$ageClass != ""'>
				<!-- assert ( $includeRacerClass == false() ) -->
				<xsl:value-of select="$ageClass"/>
			</xsl:when>
			<xsl:when test="$includeRacerClass">
				<xsl:value-of select="$racer/class"/>
			</xsl:when>
		</xsl:choose>
	</xsl:variable>
	<xsl:variable name="hasCA" select="string-length($classAnnotation) &gt; 0"/>

	<xsl:variable name="affiliationAnnotation">
		<xsl:choose>
			<xsl:when test="$includeRacerAffiliation and $racer/affiliation">
				<xsl:value-of select="$racer/affiliation/@abbrev"/>
			</xsl:when>
		</xsl:choose>
	</xsl:variable>
	<xsl:variable name="hasAA" select="string-length($affiliationAnnotation) &gt; 0"/>

	<xsl:if test="$hasCA or $hasAA">
		<xsl:text> (</xsl:text>
		<xsl:if test="$hasCA ">
			<xsl:value-of select="$classAnnotation"/>
		</xsl:if>
		<xsl:if test="$hasAA">
			<xsl:if test="$hasCA ">
				<xsl:text>, </xsl:text>
			</xsl:if>
			<xsl:value-of select="$affiliationAnnotation"/>
		</xsl:if>
		<xsl:text>)</xsl:text>
	</xsl:if>

</xsl:template>

<xsl:template name="positionString">
	<xsl:value-of select="position"/>
	<xsl:if test="position[@tie='yes']">
		<xsl:text>(t)</xsl:text>
	</xsl:if>
</xsl:template>

<xsl:template name="seriesStartPositionString">
<xsl:param name="posnAttr"/>

	<!-- series start result encodes tie indicator in the posn attr string -->
	<!--####Ver1 WAS: <xsl:variable name="posn" select="number($posnAttr)"/>-->
	<xsl:variable name="posn">
		<xsl:choose>
			<xsl:when test="contains($posnAttr, '(t)')">
				<xsl:value-of select="number(substring-before($posnAttr, '(t)'))" />
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="number($posnAttr)" />
			</xsl:otherwise>
		</xsl:choose>
	</xsl:variable>

	<xsl:choose>
		<xsl:when test="$posn = -999">
			<!-- ability class bonus in age class standings -->
			<xsl:text>OP</xsl:text>  <!-- OP for FW, SS for Intermountain... -->
		</xsl:when>
		<xsl:when test="$posn &lt; 0">
			<!-- special split-courses case: flag Group D men on separate course in overall men's standings -->
			<xsl:text>X-</xsl:text>  <!--FW used "X-" prefix; IMD uses *..* wrapper chars -->
			<xsl:value-of select="substring($posnAttr, 2)"/>  <!-- drop the leading minus char-->
			<!--<xsl:text>*</xsl:text>-->  <!--FW used "X-" prefix; IMD used *..* wrapper chars -->
		</xsl:when>
		<xsl:otherwise>
			<xsl:value-of select="$posnAttr"/>
		</xsl:otherwise>
	</xsl:choose>
</xsl:template>

<xsl:template name="composeRacerScoreEntry">
<xsl:param name="score"/>
<xsl:param name="valueIfZero" select="'0'"/>
	<xsl:choose>
		<xsl:when test="$score='0'">
			<xsl:value-of select="$valueIfZero"/>
		</xsl:when>
		<xsl:otherwise>
			<xsl:value-of select="$score"/>
		</xsl:otherwise>
	</xsl:choose>
	<!-- annotate an handicapped time with the actual time from which it was computed -->
	<xsl:if test="$score/@time">
		<xsl:text> [</xsl:text>
		<xsl:value-of select="$score/@time"/>
		<xsl:text>]</xsl:text>
	</xsl:if>
</xsl:template>

<xsl:template name="composeSeriesStartFinishEntry">
<xsl:param name="racerResult"/>
<xsl:param name="racer"/>

	<xsl:value-of select="$racerResult/@nStarts"/>
	<xsl:text>(</xsl:text><xsl:value-of select="$racerResult/@nFinishes"/><xsl:text>)</xsl:text>

	<!-- flag racers who competed in multiple categories during the series-->
	<xsl:if test="$racerResult/@multipleResults='yes'">
		<!-- assert ( $racer/@nStarts and $racer/@nStarts &gt; $racerResult/@nStarts ) -->
		<xsl:text>*</xsl:text>
	</xsl:if>

</xsl:template>

<xsl:template name="composeSeriesStartEntry">
<xsl:param name="racerResult"/>
<xsl:param name="racer"/>

	<xsl:value-of select="$racerResult/@nStarts"/>

	<!-- flag racers who competed in multiple categories during the series-->
	<xsl:if test="$racerResult/@multipleResults='yes'">
		<!-- assert ( $racer/@nStarts and $racer/@nStarts &gt; $racerResult/@nStarts ) -->
		<xsl:text>*</xsl:text>  <!-- flag a multi-class racer -->
	</xsl:if>

</xsl:template>

<xsl:template name="composeSeriesFinishEntry">
<xsl:param name="racerResult"/>
<xsl:param name="racer"/>

	<xsl:value-of select="$racerResult/@nFinishes"/>

	<!-- flag racers who competed in multiple categories during the series-->
	<xsl:if test="$racerResult/@multipleResults='yes' and $racer/@nFinishes &gt; $racerResult/@nFinishes">
		<!-- assert ( $racer/@nFinishes and $racer/@nFinishes &gt;= $racerResult/@nFinishes ) -->
		<xsl:text>*</xsl:text>  <!-- flag a multi-class racer -->
	</xsl:if>

</xsl:template>

</xsl:stylesheet>
