<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
	version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
	xmlns:fo="http://www.w3.org/1999/XSL/Format" >

<!--
 ==============================================================================
 == PDF Race Results Report Generation
 ==============================================================================
 -->

<xsl:import href="scoringutils.xsl"/>
<xsl:import href="scoringutils-pdf.xsl"/>

<xsl:output method="xml"/>  <!-- this transformation emits XML formatting to feed it into the FOP processor -->


<!--
 ==============================================================================
 == client parameters and template processing options
 ==============================================================================
 -->

<!-- default standings organization is by class -->
<xsl:param name="organizationType" select="'ByClass'"/>

<!-- optional racer name annotations - specify yes to enable -->
<xsl:param name="includeRacerClass" select="'no'"/>
<xsl:param name="includeRacerAffiliation" select="'no'"/>

<!-- following not currently used in HTML reports (just for text) -->
<xsl:param name="includeRacePoints" select="'yes'"/>

<!-- debug/trace options -->
<xsl:param name="DEBUG" select="'no'"/>


<!--
 ==============================================================================
 == global variables from ACES_RaceScoring
 ==============================================================================
 -->

<xsl:variable name="scoringRoot" select="/ACES_RaceScoring"/>

<xsl:variable name="theRace" select="$scoringRoot/race[1]"/>

<xsl:variable name="reportTitle">
	<xsl:call-template name="raceDescription">
		<xsl:with-param name="race" select="$theRace"/>
	</xsl:call-template>
</xsl:variable>

<xsl:variable name="standingsDescription">
	<xsl:text>Standings </xsl:text>
	<xsl:value-of select="$organizationDescription"/>
</xsl:variable>

<xsl:variable name="allCompetitors" 
	select="$scoringRoot/competitors/competitor"/>

<xsl:variable name="standings" 
	select="$scoringRoot/standings[@name=$organizationType]"/>
<xsl:variable name="organizationDescription" select="$standings/description"/>

<xsl:variable name="nRuns" select="$theRace/nRuns"/>


<!--
 ==============================================================================
 == report formatting styles
 ==============================================================================
 -->

<!-- ########### TO DO: Define attribute sets for formatting styles ########### -->


<!--
 ==============================================================================
 == root template: ACES_RaceScoring -> race standings report
 ==============================================================================
 -->

<xsl:template match="/">

	<!-- correctness checking (?overzealous?!) - this template isn't really strictly necessary -->
	<xsl:if test="count(ACES_RaceScoring) != 1">
		<xsl:message terminate="yes">****FATAL ERROR: cannot process &lt;<xsl:value-of select="name(child::*[1])"/>&gt; - &lt;ACES_RACEScoring&gt; scoring data required</xsl:message>
	</xsl:if>
	<xsl:if test="count($standings) != 1">
		<xsl:message terminate="yes">***FATAL ERROR: no <xsl:value-of select="$organizationType"/> &lt;standings&gt; available for race <xsl:value-of select="$theRace/raceID"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>
	<xsl:if test="not( $nRuns=1 or $nRuns=2)">
		<xsl:message terminate="yes">***FATAL ERROR: race results must have data for either 1 or 2 runs (<xsl:value-of select="$nRuns"/> runs not supported)"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>

	<xsl:apply-templates select="ACES_RaceScoring"/>

</xsl:template>


<!--
 ==============================================================================
 == ACES_RaceScoring -> race standings report
 ==============================================================================
 -->

<xsl:template match="ACES_RaceScoring">

    <fo:root >

        <!-- page layout specification -->
        <fo:layout-master-set>
            <fo:simple-page-master
                master-name="standardPage"
                page-width="8.5in"
                page-height="11in"
				>
                <fo:region-body region-name="xsl-region-body" 
                    margin="0.7in"
                    padding="6pt" />
                <fo:region-before region-name="xsl-region-before" 
                    extent="0.7in"
                    display-align="before" />
                <fo:region-after region-name="xsl-region-after" 
                    extent="0.7in"
                    display-align="after" />
            </fo:simple-page-master>
        </fo:layout-master-set>

        <!-- text flow specifications -->
        <fo:page-sequence master-reference="standardPage">
            <fo:static-content flow-name="xsl-region-before">
				<fo:block
            		font-family="Times"
					font-size="10pt"
		            font-weight="bold"
					font-style="italic"
					space-before="24pt"
					space-after="6pt"
					margin-right="0.7in"
					text-align="right"
					>
					<xsl:value-of select="$reportTitle"/>
				</fo:block>
            </fo:static-content>
            <fo:flow flow-name="xsl-region-body"
            		font-family="serif"
		            text-align="left"
					>
                <xsl:call-template name="generatePageHeader"/>
                <xsl:call-template name="generateReportHeader"/>
                <xsl:call-template name="generateReportBody"/>
                <xsl:call-template name="generateReportFooter"/>
                <xsl:call-template name="generatePageFooter"/>
            </fo:flow>
<!--#########WHAT TO DO???############
            <fo:static-content flow-name="xsl-region-after">
            </fo:static-content>
####################-->
        </fo:page-sequence>

    </fo:root>

</xsl:template>

<xsl:template name="generatePageHeader">
	<!-- div.id="pageHeader" -->
    <fo:block
            font-size="18pt"
            font-weight="bold"
            space-before="0pt"
            space-after="12pt"
            >
        <xsl:value-of select="$reportTitle"/>
    </fo:block>
</xsl:template>

<xsl:template name="generatePageFooter">
	<!-- div.id="pageFooter" -->
    <fo:block
            font-size="10pt"
            font-style="italic"
            space-before="12pt"
            space-after="0pt"
            >
        <xsl:text>Produced by </xsl:text>
        <xsl:value-of select="$scoringRoot/auxInfo/softwareDescription"/>
    </fo:block>
</xsl:template>

<!--
 ==============================================================================
 == Report header section
 ==============================================================================
 -->

<xsl:template name="generateReportHeader">

	<xsl:variable name="competitorSummary" 
		select="$standings/standingsSummary/competitorSummary"/>

	<!-- div.id="reportHeader" -->

	<!-- class="subTitle" -->
    <fo:block
		font-size="12pt"
		font-weight="bold"
		space-before="0pt"
		space-after="0pt"
		>
		<xsl:text>Race Summary</xsl:text>
    </fo:block>

    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:value-of select="$competitorSummary[@category='all']/@nStarters"/><xsl:text> competitors (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> women)</xsl:text>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:value-of select="$competitorSummary[@category='all']/@nQualifiers"/><xsl:text> finishers (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nQualifiers"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nQualifiers"/><xsl:text> women)</xsl:text>
    </fo:block>

</xsl:template>


<!--
 ==============================================================================
 == Report footer section
 ==============================================================================
 -->

<xsl:template name="generateReportFooter">

	<xsl:variable name="classes">
		<xsl:call-template name="classesDescription">
			<xsl:with-param name="classes" select="race/classes"/>
		</xsl:call-template>
	</xsl:variable>

	<!-- div.id="reportFooter" -->

	<!-- class="subTitle" -->
    <fo:block
		font-size="12pt"
		font-weight="bold"
		space-before="12pt"
		space-after="0pt"
		>
		<xsl:text>Scoring Details</xsl:text>
    </fo:block>

    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
		<xsl:text>Race points: F=</xsl:text><xsl:value-of select="scoring/fValue"/>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
		<xsl:text>Classes: </xsl:text><xsl:value-of select="$classes"/>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
		<xsl:text>Courses: </xsl:text><xsl:value-of select="race/courses"/>
    </fo:block>

</xsl:template>

<!--
 ==============================================================================
 == Report body
 ==============================================================================
 -->

<xsl:template name="generateReportBody">

	<xsl:if test="$DEBUG='yes'">
		<xsl:message>***PROCESSING &lt;standings&gt; organization <xsl:value-of select="$organizationType"/></xsl:message>
	</xsl:if>

	<!-- div.id="reportBody" -->

	<!-- FOP 0.20 doesn't support the table caption element:
	<fo:table-and-caption>
		<fo:table-caption text-align="left">
			<fo:block font-weight="bold" font-size="12pt">
			<xsl:value-of select="$standingsDescription"/>
			</fo:block>
		</fo:table-caption>
	-->
		<fo:block font-weight="bold" font-size="12pt" space-before="12pt" space-after="0pt">
			<xsl:value-of select="$standingsDescription"/>
		</fo:block>
		<fo:table inline-progression-dimension="6in" table-layout="fixed"> 
			<!--class="standings"-->
			<!-- FOP 0.20 requires column width specifications -->
			<xsl:choose>
				<xsl:when test="$nRuns=1">
					<fo:table-column column-width="0.4in"/>
					<fo:table-column column-width="2.5in"/>
					<fo:table-column column-width="0.75in"/>
					<fo:table-column column-width="0.75in"/>
				</xsl:when>
				<xsl:when test="$nRuns=2">
					<fo:table-column column-width="0.4in"/>
					<fo:table-column column-width="2.5in"/>
					<fo:table-column column-width="0.75in"/>
					<fo:table-column column-width="0.85in"/>
					<fo:table-column column-width="0.75in"/>
					<fo:table-column column-width="0.75in"/>
				</xsl:when>
				<xsl:otherwise>
					<!-- Assert.shouldNotBe( detected in entry processing) -->
					<xsl:message terminate="yes">***FATAL ERROR: Unsupported nRuns: <xsl:value-of select="$nRuns"/>"/></xsl:message>
					<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
				</xsl:otherwise>
			</xsl:choose>
			<fo:table-body>
			<xsl:apply-templates select="$standings"/>
			</fo:table-body>
		</fo:table>
	<!-- FOP 0.20 doesn't support the table caption element:
	</fo:table-and-caption>
	-->

</xsl:template>

<xsl:template match="standings">

	<xsl:variable name="nGroups" select="count(groupStandings)"/>
	<xsl:variable name="includeCategoryHeaders" select="not (@singleCategory='yes')"/>  <!-- and nGroups > 1 -->

	<xsl:variable name="nColumns">
		<xsl:choose>
			<xsl:when test="$nRuns=2">
				<xsl:text>6</xsl:text>
			</xsl:when>
			<xsl:otherwise>
				<xsl:text>4</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:variable>

	<xsl:if test="not($includeCategoryHeaders)">
		<fo:table-row font-size="10pt" font-weight="bold" keep-with-next="always"> <!--class="standingsHeader"-->
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Place</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Competitor</fo:block></fo:table-cell>
			<xsl:if test="$nRuns=2">
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">First Run</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Second Run</fo:block></fo:table-cell>
			</xsl:if>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Time</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Race Points</fo:block></fo:table-cell>
		</fo:table-row>
	</xsl:if>

	<xsl:for-each select="groupStandings">
		<xsl:call-template name="processResultGroup">
			<xsl:with-param name="includeCategoryHeaders" select="$includeCategoryHeaders"/>
			<xsl:with-param name="nColumns" select="$nColumns"/>
		</xsl:call-template>
	</xsl:for-each>

</xsl:template>

<xsl:template name="processResultGroup">
<xsl:param name="includeCategoryHeaders"/>
<xsl:param name="nColumns"/>

	<xsl:if test="$includeCategoryHeaders">
		<fo:table-row font-size="10pt" font-weight="bold" keep-with-next="always">  <!--class="groupHeader"-->
			<fo:table-cell number-columns-spanned="2"><fo:block space-before="6pt" space-after="3pt">
			<xsl:value-of select="description"/>
			</fo:block></fo:table-cell>
			<xsl:if test="$nRuns=2">
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">First Run</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Second Run</fo:block></fo:table-cell>
			</xsl:if>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Time</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Race Points</fo:block></fo:table-cell>
		</fo:table-row>
	</xsl:if>

	<xsl:for-each select="racerResult">
		<xsl:call-template name="processRacerResult"/>
	</xsl:for-each>

	<!-- for race results, we don't care about optional category summary - rare to have ties in the group -->

</xsl:template>

<xsl:template name="processRacerResult">

	<xsl:variable name="racerID" select="@racerID"/>
	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>

	<xsl:variable name="status" select="$racer/@status"/>
	<xsl:variable name="racerName">
		<xsl:call-template name="composeRacerNameEntry">
			<xsl:with-param name="racer" select="$racer"/>
			<xsl:with-param name="ageClass" select="@ageClass"/>  <!-- optional attr in OP/SS class standings -->
			<xsl:with-param name="includeRacerClass" 
				select="$includeRacerClass='yes'"/>
			<xsl:with-param name="includeRacerAffiliation" 
				select="$includeRacerAffiliation='yes'"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:if test="$status!='DNS'">
	<fo:table-row font-size="10pt">  <!--class="racerEntry"-->
		<fo:table-cell><fo:block> <!--class="position"-->
		<xsl:choose>
			<xsl:when test="$status='QLF'">
				<xsl:call-template name="positionString"/>
			</xsl:when>
			<xsl:otherwise>
				<xsl:text>-</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
		</fo:block></fo:table-cell>
		<fo:table-cell><fo:block> <!--class="name"-->
		<xsl:value-of select="$racerName"/>
		</fo:block></fo:table-cell>
		<xsl:if test="$nRuns=2">
			<fo:table-cell><fo:block> <!--class="run1"-->
			<xsl:value-of select="$racer/run1"/>
			</fo:block></fo:table-cell>
			<fo:table-cell><fo:block> <!--class="run2"-->
			<xsl:value-of select="$racer/run2"/>
			</fo:block></fo:table-cell>
		</xsl:if>
		<fo:table-cell><fo:block> <!--class="result"-->
		<xsl:value-of select="$racer/result"/>
		</fo:block></fo:table-cell>
		<xsl:choose>
			<xsl:when test="$status='QLF'">
				<fo:table-cell><fo:block> <!--class="racePoints"-->
				<xsl:value-of select="racePoints"/>
				</fo:block></fo:table-cell>
			</xsl:when>
			<xsl:otherwise>
				<xsl:call-template name="emptyTableCell"/>
			</xsl:otherwise>
		</xsl:choose>
	</fo:table-row>
	</xsl:if>

</xsl:template>

</xsl:stylesheet>
	
