<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
	version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

<!--
 ==============================================================================
 == HTML Series Standings Report Generation
 ==============================================================================
 -->

<xsl:import href="scoringutils.xsl"/>
<xsl:import href="scoringutils-html.xsl"/>

<xsl:output method="html" 
	version="4.1" 
	doctype-public="-//W3C//DTD HTML 4.01//EN"
	indent="yes"/>


<!--
 ==============================================================================
 == client parameters and template processing options
 ==============================================================================
 -->

<!-- default standings organization is by class -->
<xsl:param name="organizationType" select="'ByClass'"/>

<!-- optional racer name annotations - specify yes to enable -->
<xsl:param name="includeRacerAffiliation" select="'no'"/>
<xsl:param name="includeRacerClass" select="'no'"/>

<!-- debug/trace options -->
<xsl:param name="DEBUG" select="'no'"/>


<!--
 ==============================================================================
 == global variables from ACES_SeriesScoring
 ==============================================================================
 -->

<xsl:variable name="scoringRoot" select="/ACES_SeriesScoring"/>

<xsl:variable name="theEvent" select="$scoringRoot/event[1]"/>

<xsl:variable name="reportTitle">
	<xsl:call-template name="eventDescription">
		<xsl:with-param name="event" select="$theEvent"/>
	</xsl:call-template>
</xsl:variable>

<xsl:variable name="standingsDescription">
	<xsl:value-of select="$theEvent/standingsType"/>
	<xsl:text> Standings </xsl:text>
	<xsl:value-of select="$organizationDescription"/>
</xsl:variable>

<xsl:variable name="allCompetitors" 
	select="$scoringRoot/competitors/competitor"/>

<xsl:variable name="standings" 
	select="$scoringRoot/standings[@name=$organizationType]"/>
<xsl:variable name="organizationDescription" select="$standings/description"/>

<xsl:variable name="seriesRaces" select="$theEvent/races/race"/>
<xsl:variable name="nRaces" select="count($seriesRaces)"/>


<!--
 ==============================================================================
 == root template: ACES_SeriesScoring-> series standings report
 ==============================================================================
 -->

<!-- note that this isn't actually necessary -->
<xsl:template match="/">

	<!-- correctness checking (?overzealous?!) - this template isn't really strictly necessary -->
	<xsl:if test="count(ACES_SeriesScoring) != 1">
		<xsl:message terminate="yes">****FATAL ERROR: cannot process &lt;<xsl:value-of select="name(child::*[1])"/>&gt; - &lt;ACES_SeriesScoring&gt; scoring data required</xsl:message>
	</xsl:if>
	<xsl:if test="count($standings) != 1">
		<xsl:message terminate="yes">***FATAL ERROR: no <xsl:value-of select="$organizationType"/> &lt;standings&gt; available for event<xsl:value-of select="$theEvent/name"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>

	<xsl:apply-templates select="ACES_SeriesScoring"/>

</xsl:template>


<!--
 ==============================================================================
 == ACES_SeriesScoring -> series/event standings report
 ==============================================================================
 -->

<xsl:template match="ACES_SeriesScoring">

<html>
<head>

<title><xsl:value-of select="$reportTitle"/> - <xsl:value-of select="$organizationDescription"/></title>

<link rel="stylesheet" type="text/css" href="seriesscoring.css"/>
<style type="text/css">
/* embedded document styles */
</style>
</head>

<xsl:comment>============================== DOCUMENT BODY ===========================</xsl:comment>
<body>

<xsl:call-template name="generatePageHeader"/>

<xsl:call-template name="generateReportHeader"/>
<xsl:call-template name="generateReportBody"/>
<xsl:call-template name="generateReportFooter"/>

<xsl:call-template name="generatePageFooter"/>

</body>
</html>

</xsl:template>

<xsl:template name="generatePageHeader">
<div id="pageHeader">
	<h1><xsl:value-of select="$reportTitle"/></h1>
</div>
</xsl:template>

<xsl:template name="generatePageFooter">
<div id="pageFooter">
	<p>
		<xsl:text>Produced by </xsl:text>
		<xsl:value-of select="$scoringRoot/auxInfo/softwareDescription"/>
	</p>
</div>
</xsl:template>


<!--
 ==============================================================================
 == Report header section
 ==============================================================================
 -->

<xsl:template name="generateReportHeader">

	<xsl:variable name="competitorSummary" 
		select="$standings/standingsSummary/competitorSummary"/>
	<xsl:variable name="competitorStarts"
		select="$standings/standingsSummary/competitorStarts"/>
	<xsl:variable name="raceStarts"
		select="$standings/standingsSummary/raceStarts"/>	

<div id="reportHeader">

	<p>
	<span class="subTitle"><xsl:value-of select="$theEvent/standingsType"/> Standings Summary</span>
	<br/><xsl:text>&#xA;</xsl:text>
	<xsl:choose>
		<xsl:when test="$nRaces=1">
			<xsl:text>Single-race event scoring.</xsl:text>
		</xsl:when>
		<xsl:otherwise>
			<xsl:text></xsl:text><xsl:value-of select="$nRaces"/><xsl:text> races scored.</xsl:text>
		</xsl:otherwise>
	</xsl:choose>
	<br/><xsl:text>&#xA;</xsl:text>
	<xsl:text>Scoring policy: </xsl:text>
	<!-- DROP: <xsl:value-of select="$organizationType"/><xsl:text> - </xsl:text> -->
	<xsl:value-of select="scoring/scoringPolicy"/><xsl:text>.</xsl:text>
	</p>

	<!-- Season scoring: total number of starts, ave starts per racer, starts per discipline -->
	<xsl:if test="$raceStarts">
		<!-- assert( "$theEvent/standingsType='Season'" ) -->
		<xsl:variable name="allRaceStarts" select="$raceStarts[@category='all']"/>
		<p>
		<xsl:text>Total number of starts: </xsl:text><xsl:value-of select="$allRaceStarts/@total"/>
		<br/><xsl:text>&#xA;</xsl:text>
		<xsl:text>Average starters per race: </xsl:text><xsl:value-of select="$allRaceStarts/@ave"/>
		<br/><xsl:text>&#xA;</xsl:text>
		<xsl:for-each select="$raceStarts[@category != 'all']">
			<xsl:text>Ave starters </xsl:text><xsl:value-of select="@category"/><xsl:text>: </xsl:text>
			<xsl:value-of select="@ave"/>
			<xsl:text> (</xsl:text><xsl:value-of select="@total"/><xsl:text> </xsl:text>
			<xsl:value-of select="@category"/><xsl:text> starts, </xsl:text>
			<xsl:value-of select="@nRaces"/><xsl:text> races)</xsl:text>
			<br/><xsl:text>&#xA;</xsl:text>
		</xsl:for-each>
		</p>
	</xsl:if>

	<p>
	<xsl:text>Total number of competitors: </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='all']/@nStarters"/><xsl:text> (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> women)</xsl:text>
	<br/><xsl:text>&#xA;</xsl:text>
	<xsl:text>Number of qualifiers: </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='all']/@nQualifiers"/><xsl:text> (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nQualifiers"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nQualifiers"/><xsl:text> women)</xsl:text>
	<xsl:if test="$theEvent/standingsType='Season'">
		<br/><xsl:text>&#xA;</xsl:text>
<!--VER 1 WAS:
		<xsl:text>Average number of starts per racer: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='all']/@ave"/><xsl:text> (</xsl:text>
		<xsl:value-of select="$competitorStarts[@category='M']/@ave"/><xsl:text> men, </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='F']/@ave"/><xsl:text> women)</xsl:text>
-->
		<xsl:variable name="allRacerStarts" select="$competitorStarts[@category='all']"/>
		<xsl:text>Average starts per racer: </xsl:text><xsl:value-of select="$allRacerStarts/@ave"/>
		<br/><xsl:text>&#xA;</xsl:text>
		<xsl:text>Ave starts men: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='M']/@ave"/>
		<xsl:text> (</xsl:text><xsl:value-of select="$competitorStarts[@category='M']/@total"/>
		<xsl:text> starts, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> competitors)</xsl:text>
		<br/><xsl:text>&#xA;</xsl:text>
		<xsl:text>Ave starts women: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='F']/@ave"/>
		<xsl:text> (</xsl:text><xsl:value-of select="$competitorStarts[@category='F']/@total"/>
		<xsl:text> starts, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> competitors)</xsl:text>
		<br/><xsl:text>&#xA;</xsl:text>
	</xsl:if>
	</p>

	<!-- DIVISIONS CUP SUMMARY (special-case hack, oh well) -->
	<xsl:variable name="teamResults" 
		select="$standings/standingsSummary/teamSummary"/>
	<xsl:if test="$teamResults">
		<!-- assert( "$organizationType='ByAffiliation' and policy is Divisions Cup..." ) -->
		<!-- CONSIDER: maybe put this sucker in a table and be done with it?? -->
		<table class="summary" border="1">
			<caption>Divisions Cup Standings (points per start)</caption>
			<tr class="summaryHeader">
				<td>Position</td>
				<td>Team</td>
				<td>Pts/Strt</td>
				<td># Competitors</td>
				<td># Starts</td>
				<td>Points</td>
			</tr>
			<xsl:for-each select="$teamResults">
				<tr class="summaryEntry">
				<td>
				<xsl:choose>
					<xsl:when test="@status='QLF'">
						<xsl:value-of select="@posn"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:text>-</xsl:text>
					</xsl:otherwise>
				</xsl:choose>
				</td>
				<td><xsl:value-of select="description"/></td>  <!-- @name has short name -->
				<td><xsl:value-of select="@ave"/></td>
				<td><xsl:value-of select="@nRacers"/></td>
				<td><xsl:value-of select="@nStarts"/></td>
				<td><xsl:value-of select="@total"/></td>
				</tr>
			</xsl:for-each>
		</table>
		<p><xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text></p>

	</xsl:if>

	<!-- MEDAL COUNTS SUMMARY (special-case hack, oh well) -->
	<xsl:variable name="teamMedals" 
		select="$standings/standingsSummary/teamMedals"/>
	<xsl:if test="$teamMedals">
		<!-- assert( "$organizationType='ByAffiliation' and policy is Medal Counts..." ) -->
		<!-- CONSIDER: maybe put this sucker in a table and be done with it?? -->
		<table class="summary" border="1">
			<caption>Medal Standings</caption>
			<tr class="summaryHeader">
				<td>Position</td>
				<td>Team</td>
				<td># Medals</td>
				<td>Gold</td>
				<td>Silver</td>
				<td>Bronze</td>
				<td># Competitors</td>
				<td># Medalists</td>
			</tr>
			<xsl:for-each select="$teamMedals">
				<tr class="summaryEntry">
				<td>
				<xsl:choose>
					<xsl:when test="@status='QLF'">
						<xsl:value-of select="@posn"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:text>-</xsl:text>
					</xsl:otherwise>
				</xsl:choose>
				</td>
				<td><xsl:value-of select="description"/></td>  <!-- @name has short name -->
				<td><xsl:value-of select="@total"/></td>
				<td><xsl:value-of select="@gold"/></td>
				<td><xsl:value-of select="@silver"/></td>
				<td><xsl:value-of select="@bronze"/></td>
				<td><xsl:value-of select="@nRacers"/></td>
				<td><xsl:value-of select="@nMedalists"/></td>
				</tr>
			</xsl:for-each>
		</table>
		<p><xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text></p>

	</xsl:if>

</div>

</xsl:template>


<!--
 ==============================================================================
 == Report footer section
 ==============================================================================
 -->

<xsl:template name="generateReportFooter">

	<xsl:variable name="classes">
		<xsl:call-template name="classesDescription">
			<xsl:with-param name="classes" select="event/classes"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:variable name="scoringElement">
		<xsl:call-template name="scoringElementDescription">
			<xsl:with-param name="scoringElementName" select="scoring/scoringElement"/>
		</xsl:call-template>
	</xsl:variable>

<div id="reportFooter">
	<p>
	<span class="subTitle">Scoring Details</span><br/>
	<xsl:text>Classes: </xsl:text><xsl:value-of select="$classes"/>
	<br/><xsl:text>&#xA;</xsl:text>
	<xsl:text>Scoring element: </xsl:text><xsl:value-of select="$scoringElement"/>
	<br/><xsl:text>&#xA;</xsl:text>
	<xsl:text>Scoring policy: </xsl:text><xsl:value-of select="scoring/scoringPolicy"/>
	</p>
	<p>
	<span class="subTitle">Scored Races</span>
	<br/><xsl:text>&#xA;</xsl:text>
	<xsl:for-each select="$seriesRaces">
		<xsl:value-of select="position()"/>
		<xsl:text>. </xsl:text>
		<xsl:call-template name="raceDescription">
			<xsl:with-param name="race" select="."/>
		</xsl:call-template>
		<xsl:text> - </xsl:text>
		<xsl:value-of select="courses"/>
		<br/><xsl:text>&#xA;</xsl:text>
	</xsl:for-each>
	</p>

	<xsl:variable name="multiCategoryRacers" 
		select="$standings/groupStandings/racerResult[@multipleResults='yes']"/>
			<!-- WAS: $allCompetitors[@nStarts] -->
	<xsl:if test="$multiCategoryRacers">
		<xsl:choose>
			<xsl:when test="$organizationType='ByClass'">
				<p>
				<span class="subTitle">Racers with starts in both Open seed and age class</span>
				<br/><xsl:text>&#xA;</xsl:text>
				<xsl:text>Racer name - total number of starts (total number of finishes)</xsl:text>
<!-- ####### TO DO [DJL 16-Nov-2002] ########### -->
<br/><xsl:text>&#xA;</xsl:text>
<xsl:text>***NOTE: Need to fix sorting!!</xsl:text>
				<br/>
				<xsl:call-template name="generateMultiClassRacersReport">
					<xsl:with-param name="multiClassRacers" select="$multiCategoryRacers"/>
					<xsl:with-param name="index" select="1"/>
					<xsl:with-param name="prevAgeClass" select="'NONE'"/>
				</xsl:call-template>
				</p>
			</xsl:when>
			<xsl:otherwise>
				<!-- //probably overzealous; if this is really serious, should detect long before now, eh??
				<xsl:message terminate="yes">***ERROR: should not have racers in multiple categories!!</xsl:message>
				-->
				<p><i><b>***WARNING:</b> <xsl:value-of select="count($multiCategoryRacers)"/> racers have starts in multiple categories - should not be!!!</i></p>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:if>

</div>

</xsl:template>

<xsl:template name="generateMultiClassRacersReport">
<xsl:param name="multiClassRacers"/>
<xsl:param name="index"/>
<xsl:param name="prevAgeClass"/>

	<xsl:variable name="racerResult" select="$multiClassRacers[$index]"/>
	<xsl:variable name="racerID" select="$racerResult/@racerID"/>
	<xsl:variable name="currentClass" select="$racerResult/../@name"/>  <!-- groupStandings category name attr -->

	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>
	<xsl:variable name="racerAgeClass" select="$racer/class"/>

	<!-- report on age clases; skip ability class results, they're the dup guys -->
	<xsl:if test="$racerAgeClass=$currentClass">

		<!-- emit separator and category header between age classes -->
		<xsl:if test="$racerAgeClass != $prevAgeClass">
			<!-- ISSUE: Just have name of the class at this point; we've lost its (category) description... -->
			<br/><xsl:text>&#xA;</xsl:text>
			<xsl:value-of select="$racerAgeClass"/>
			<br/><xsl:text>&#xA;</xsl:text>
		</xsl:if>

		<xsl:value-of select="$racer/lastName"/>, <xsl:value-of select="$racer/firstName"/>
		<xsl:text> - </xsl:text>
		<xsl:value-of select="$racer/@nStarts"/>
		<xsl:text> (</xsl:text>
		<xsl:value-of select="$racer/@nFinishes"/>
		<xsl:text>)</xsl:text>
		<br/><xsl:text>&#xA;</xsl:text>
	</xsl:if>

	<xsl:if test="$index != count($multiClassRacers)">
		<xsl:call-template name="generateMultiClassRacersReport">
			<xsl:with-param name="multiClassRacers" select="$multiClassRacers"/>
			<xsl:with-param name="index" select="$index + 1"/>
			<xsl:with-param name="prevAgeClass" select="$currentClass"/>
		</xsl:call-template>
	</xsl:if>

</xsl:template>


<!--
 ==============================================================================
 == Report body
 ==============================================================================
 -->

<xsl:template name="generateReportBody">

	<xsl:if test="$DEBUG='yes'">
		<xsl:message>***PROCESSING &lt;standings&gt; organization <xsl:value-of select="$organizationType"/></xsl:message>
	</xsl:if>

<div id="reportBody">
	<table class="standings" border="1">
		<caption><xsl:value-of select="$standingsDescription"/></caption>
		<tbody>
		<tr class="standingsHeader">
			<td class="position">Position</td>
			<td class="name">Competitor</td>
			<td class="score">
				<xsl:call-template name="scoringElementLabel">
					<xsl:with-param name="scoringElementName" select="scoring/scoringElement"/>
				</xsl:call-template>
			</td>
			<!-- WAS: <td class="sfCount">St(Fn)</td> -->
			<td class="sfCount">St</td>
			<td class="sfCount">Fn</td>
			<xsl:call-template name="generateRaceColumnLabels"/>
		</tr>
		<xsl:apply-templates select="$standings"/>
		</tbody>
	</table>
</div>

</xsl:template>

<xsl:template match="standings">

	<xsl:variable name="nGroups" select="count(groupStandings)"/>
	<xsl:variable name="includeCategoryHeaders" select="not (@singleCategory='yes')"/>  <!-- and nGroups > 1 -->

	<xsl:variable name="nColumns" select="5 + $nRaces"/>

	<xsl:for-each select="groupStandings">
		<xsl:call-template name="processResultGroup">
			<xsl:with-param name="includeCategoryHeaders" select="$includeCategoryHeaders"/>
			<xsl:with-param name="nColumns" select="$nColumns"/>
		</xsl:call-template>
	</xsl:for-each>

</xsl:template>

<xsl:template name="processResultGroup">
<xsl:param name="includeCategoryHeaders"/>
<xsl:param name="nColumns"/>

	<xsl:if test="$includeCategoryHeaders">
		<xsl:call-template name="emptyTableRow">
			<xsl:with-param name="nColumns" select="$nColumns"/>
		</xsl:call-template>
		<tr class="groupHeader">
			<xsl:call-template name="emptyTableCell"/>
			<td class="name"><xsl:value-of select="description"/></td>
			<xsl:call-template name="emptyTableCell"/>
			<xsl:call-template name="emptyTableCell"/>
			<xsl:call-template name="emptyTableCell"/>
			<xsl:call-template name="generateRaceColumnLabels"/>
		</tr>
	</xsl:if>

	<xsl:for-each select="racerResult">
		<xsl:call-template name="processRacerResult"/>
	</xsl:for-each>

	<!-- optional category summary -->
	<xsl:variable name="nQualifiers" select="@nQualifiers"/>
	<xsl:if test="$nQualifiers">
		<xsl:variable name="nCategoryCompetitors" select="count(racerResult)"/>
		<tr>
		<xsl:call-template name="emptyTableCell"/>
		<td class="categorySummary">
			<xsl:text>(</xsl:text>
			<xsl:value-of select="$nQualifiers"/>
			<xsl:if test="$nQualifiers != $nCategoryCompetitors">
				<xsl:text> of </xsl:text>
				<xsl:value-of select="$nCategoryCompetitors"/>
			</xsl:if>
			<xsl:text> competitors</xsl:text>
			<xsl:if test="$nQualifiers != $nCategoryCompetitors">
				<xsl:text> qualified</xsl:text>
			</xsl:if>
			<xsl:text>)</xsl:text>
		</td>
		<xsl:call-template name="emptyTableCell">
			<xsl:with-param name="count" select="$nColumns - 2"/>
		</xsl:call-template>
		</tr>
	</xsl:if>

</xsl:template>

<xsl:template name="processRacerResult">

	<xsl:variable name="racerID" select="@racerID"/>
	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>

	<xsl:variable name="status" select="@status"/>
	<xsl:variable name="racerName">
		<xsl:call-template name="composeRacerNameEntry">
			<xsl:with-param name="racer" select="$racer"/>
			<xsl:with-param name="ageClass" select="@ageClass"/>  <!-- optional attr in OP/SS class standings -->
			<xsl:with-param name="includeRacerClass" 
				select="$includeRacerClass='yes'"/>
			<xsl:with-param name="includeRacerAffiliation" 
				select="$includeRacerAffiliation='yes'"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:if test="$status!='DNS'">
	<tr class="racerEntry">
		<xsl:choose>
			<xsl:when test="$status='QLF'">
				<td class="position"><xsl:call-template name="positionString"/></td>
				<td class="name"><xsl:value-of select="$racerName"/></td>
				<td class="score">
					<xsl:call-template name="composeRacerScoreEntry">
						<xsl:with-param name="score" select="score"/>
						<xsl:with-param name="valueIfZero" select="'-'"/>
					</xsl:call-template>
				</td>
			</xsl:when>
			<xsl:otherwise>
				<td class="position"><xsl:text>-</xsl:text></td>
				<td class="name"><xsl:value-of select="$racerName"/></td>
				<!-- ISSUE: report score if present, even though not qualified?? -->
				<xsl:call-template name="emptyTableCell"/>
			</xsl:otherwise>
		</xsl:choose>
		<td class="sfCount">
			<xsl:call-template name="composeSeriesStartEntry">
				<xsl:with-param name="racerResult" select="."/>
				<xsl:with-param name="racer" select="$racer"/>
			</xsl:call-template>
		</td>
		<td class="sfCount">
			<xsl:call-template name="composeSeriesFinishEntry">
				<xsl:with-param name="racerResult" select="."/>
				<xsl:with-param name="racer" select="$racer"/>
			</xsl:call-template>
		</td>
		<xsl:call-template name="generateRaceScoreEntries">
			<xsl:with-param name="racerResult" select="."/>
		</xsl:call-template>
	</tr>
	</xsl:if>

</xsl:template>

<xsl:template name="generateRaceColumnLabels">
	<xsl:variable name="useLineBreak" select="$nRaces &gt; 6"/>
	<xsl:for-each select="$seriesRaces">
		<td class="rsScore">
			<xsl:value-of select="location/@name"/>
			<!-- HEURISTIC EXPERIMENT -->
			<xsl:choose>
				<xsl:when test="$useLineBreak">
					<br/>
				</xsl:when>
				<xsl:otherwise>
					<xsl:text> </xsl:text>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:value-of select="discipline"/>
		</td>
	</xsl:for-each>
</xsl:template>

<xsl:template name="generateRaceScoreEntries">
<xsl:param name="racerResult"/>
<xsl:param name="index" select="1"/>

	<xsl:variable name="raceScore" select="$racerResult/rsScore[@ri=$index]"/>

	<xsl:choose>
		<xsl:when test="$raceScore">
			<td class="rsScore">
			<!-- #### TO DO (someday): if $racerResult/@counted='no' -> flag excluded score ####-->
			<xsl:value-of select="$raceScore"/>
			<xsl:if test="$raceScore/@posn">
				<xsl:text> (</xsl:text>
				<xsl:call-template name="seriesStartPositionString">
					<xsl:with-param name="posnAttr" select="$raceScore/@posn"/>
				</xsl:call-template>
				<xsl:text>)</xsl:text>
			</xsl:if>
			</td>
		</xsl:when>
		<xsl:otherwise>
			<xsl:call-template name="emptyTableCell"/>
		</xsl:otherwise>
	</xsl:choose>

	<xsl:if test="not ($index = $nRaces)">
		<xsl:call-template name="generateRaceScoreEntries">
			<xsl:with-param name="racerResult" select="$racerResult"/>
			<xsl:with-param name="index" select="$index + 1"/>
		</xsl:call-template>
	</xsl:if>

</xsl:template>

</xsl:stylesheet>
