<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet
	version="1.0"
	xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
	xmlns:fo="http://www.w3.org/1999/XSL/Format" >

<!--
 ==============================================================================
 == PDF Series Standings Report Generation
 ==============================================================================
 -->

<xsl:import href="scoringutils.xsl"/>
<xsl:import href="scoringutils-pdf.xsl"/>

<xsl:output method="xml"/>  <!-- this transformation emits XML formatting to feed it into the FOP processor -->

<!--
 ==============================================================================
 == client parameters and template processing options
 ==============================================================================
 -->

<!-- default standings organization is by class -->
<xsl:param name="organizationType" select="'ByClass'"/>

<!-- optional racer name annotations - specify yes to enable -->
<xsl:param name="includeRacerAffiliation" select="'no'"/>
<xsl:param name="includeRacerClass" select="'no'"/>

<!-- debug/trace options -->
<xsl:param name="DEBUG" select="'no'"/>


<!--
 ==============================================================================
 == global variables from ACES_SeriesScoring
 ==============================================================================
 -->

<xsl:variable name="scoringRoot" select="/ACES_SeriesScoring"/>

<xsl:variable name="theEvent" select="$scoringRoot/event[1]"/>

<xsl:variable name="reportTitle">
	<xsl:call-template name="eventDescription">
		<xsl:with-param name="event" select="$theEvent"/>
	</xsl:call-template>
</xsl:variable>

<xsl:variable name="standingsDescription">
	<xsl:value-of select="$theEvent/standingsType"/>
	<xsl:text> Standings </xsl:text>
	<xsl:value-of select="$organizationDescription"/>
</xsl:variable>

<xsl:variable name="allCompetitors" 
	select="$scoringRoot/competitors/competitor"/>

<xsl:variable name="standings" 
	select="$scoringRoot/standings[@name=$organizationType]"/>
<xsl:variable name="organizationDescription" select="$standings/description"/>

<xsl:variable name="seriesRaces" select="$theEvent/races/race"/>
<xsl:variable name="nRaces" select="count($seriesRaces)"/>


<!--
 ==============================================================================
 == report formatting styles
 ==============================================================================
 -->

<!-- ########### TO DO: Define attribute sets for formatting styles ########### -->


<!--
 ==============================================================================
 == root template: ACES_SeriesScoring-> series standings report
 ==============================================================================
 -->

<xsl:template match="/">

	<!-- correctness checking (?overzealous?!) - this template isn't really strictly necessary -->
	<xsl:if test="count(ACES_SeriesScoring) != 1">
		<xsl:message terminate="yes">****FATAL ERROR: cannot process &lt;<xsl:value-of select="name(child::*[1])"/>&gt; - &lt;ACES_SeriesScoring&gt; scoring data required</xsl:message>
	</xsl:if>
	<xsl:if test="count($standings) != 1">
		<xsl:message terminate="yes">***FATAL ERROR: no <xsl:value-of select="$organizationType"/> &lt;standings&gt; available for event<xsl:value-of select="$theEvent/name"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>
	<xsl:if test="$nRaces &gt; 4">
		<xsl:message terminate="yes">***FATAL ERROR: series results of more than 4 races not yet supported (<xsl:value-of select="$nRaces"/>)"/></xsl:message>
		<xsl:message terminate="yes">**** PROCESSING ABORTED ****</xsl:message>
	</xsl:if>

	<xsl:apply-templates select="ACES_SeriesScoring"/>

</xsl:template>


<!--
 ==============================================================================
 == ACES_SeriesScoring -> series/event standings report
 ==============================================================================
 -->

<xsl:template match="ACES_SeriesScoring">

    <fo:root >

        <!-- page layout specification -->
        <fo:layout-master-set>
            <fo:simple-page-master
                master-name="standardPage"
                page-width="8.5in"
                page-height="11in"
				>
                <fo:region-body region-name="xsl-region-body" 
                    margin="0.7in"
                    padding="6pt" />
                <fo:region-before region-name="xsl-region-before" 
                    extent="0.7in"
                    display-align="before" />
                <fo:region-after region-name="xsl-region-after" 
                    extent="0.7in"
                    display-align="after" />
            </fo:simple-page-master>
        </fo:layout-master-set>

        <!-- text flow specifications -->
        <fo:page-sequence master-reference="standardPage">
            <fo:static-content flow-name="xsl-region-before">
				<fo:block
            		font-family="Times"
					font-size="10pt"
		            font-weight="bold"
					font-style="italic"
					space-before="24pt"
					space-after="6pt"
					margin-right="0.7in"
					text-align="right"
					>
					<xsl:value-of select="$reportTitle"/>
					<xsl:text> - </xsl:text>
					<xsl:value-of select="$organizationDescription"/>
				</fo:block>
            </fo:static-content>
            <fo:flow flow-name="xsl-region-body"
            		font-family="serif"
		            text-align="left"
					>
                <xsl:call-template name="generatePageHeader"/>
                <xsl:call-template name="generateReportHeader"/>
                <xsl:call-template name="generateReportBody"/>
                <xsl:call-template name="generateReportFooter"/>
                <xsl:call-template name="generatePageFooter"/>
            </fo:flow>
<!--#########WHAT TO DO???############
            <fo:static-content flow-name="xsl-region-after">
            </fo:static-content>
####################-->
        </fo:page-sequence>

    </fo:root>

</xsl:template>

<xsl:template name="generatePageHeader">
	<!-- div.id="pageHeader" -->
    <fo:block
            font-size="18pt"
            font-weight="bold"
            space-before="0pt"
            space-after="12pt"
            >
        <xsl:value-of select="$reportTitle"/>
    </fo:block>
</xsl:template>

<xsl:template name="generatePageFooter">
	<!-- div.id="pageFooter" -->
    <fo:block
            font-size="10pt"
            font-style="italic"
            space-before="12pt"
            space-after="0pt"
            >
        <xsl:text>Produced by </xsl:text>
        <xsl:value-of select="$scoringRoot/auxInfo/softwareDescription"/>
    </fo:block>
</xsl:template>

<!--
 ==============================================================================
 == Report header section
 ==============================================================================
 -->

<xsl:template name="generateReportHeader">

	<xsl:variable name="competitorSummary" 
		select="$standings/standingsSummary/competitorSummary"/>
	<xsl:variable name="competitorStarts"
		select="$standings/standingsSummary/competitorStarts"/>
	<xsl:variable name="raceStarts"
		select="$standings/standingsSummary/raceStarts"/>	

	<!-- div.id="reportHeader" -->

	<!-- class="subTitle" -->
    <fo:block
		font-size="12pt"
		font-weight="bold"
		space-before="0pt"
		space-after="0pt"
		>
		<xsl:value-of select="$theEvent/standingsType"/>
		<xsl:text> Standings Summary</xsl:text>
    </fo:block>

    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
		<xsl:choose>
			<xsl:when test="$nRaces=1">
				<xsl:text>Single-race event scoring.</xsl:text>
			</xsl:when>
			<xsl:otherwise>
				<xsl:value-of select="$nRaces"/><xsl:text> races scored.</xsl:text>
			</xsl:otherwise>
		</xsl:choose>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:text>Scoring policy: </xsl:text>
	<!-- DROP: <xsl:value-of select="$organizationType"/><xsl:text> - </xsl:text> -->
	<xsl:value-of select="scoring/scoringPolicy"/><xsl:text>.</xsl:text>
    </fo:block>

	<!-- Season scoring: total number of starts, ave starts per racer, starts per discipline -->
	<xsl:if test="$raceStarts">
		<!-- assert( "$theEvent/standingsType='Season'" ) -->
		<xsl:variable name="allRaceStarts" select="$raceStarts[@category='all']"/>
		<fo:block
			font-size="10pt"
			space-before="12pt"
			space-after="0pt"
			>
		<xsl:text>Total number of starts: </xsl:text><xsl:value-of select="$allRaceStarts/@total"/>
		</fo:block>
		<fo:block
			font-size="10pt"
			space-before="0pt"
			space-after="0pt"
			>
		<xsl:text>Average starters per race: </xsl:text><xsl:value-of select="$allRaceStarts/@ave"/>
    	</fo:block>
		<xsl:for-each select="$raceStarts[@category != 'all']">
		<fo:block
			font-size="10pt"
			space-before="0pt"
			space-after="0pt"
			>
			<xsl:text>Ave starters </xsl:text><xsl:value-of select="@category"/><xsl:text>: </xsl:text>
			<xsl:value-of select="@ave"/>
			<xsl:text> (</xsl:text><xsl:value-of select="@total"/><xsl:text> </xsl:text>
			<xsl:value-of select="@category"/><xsl:text> starts, </xsl:text>
			<xsl:value-of select="@nRaces"/><xsl:text> races)</xsl:text>
    	</fo:block>
		</xsl:for-each>
	</xsl:if>

    <fo:block
		font-size="10pt"
		space-before="12pt"
		space-after="0pt"
		>
		<xsl:text>Total number of competitors: </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='all']/@nStarters"/><xsl:text> (</xsl:text>
		<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> men, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> women)</xsl:text>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:text>Number of qualifiers: </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='all']/@nQualifiers"/><xsl:text> (</xsl:text>
	<xsl:value-of select="$competitorSummary[@category='M']/@nQualifiers"/><xsl:text> men, </xsl:text>
	<xsl:value-of select="$competitorSummary[@category='F']/@nQualifiers"/><xsl:text> women)</xsl:text>
    </fo:block>

	<xsl:if test="$theEvent/standingsType='Season'">
	    <fo:block
			font-size="10pt"
			space-before="12pt"
			space-after="0pt"
			>
		<xsl:variable name="allRacerStarts" select="$competitorStarts[@category='all']"/>
		<xsl:text>Average starts per racer: </xsl:text><xsl:value-of select="$allRacerStarts/@ave"/>
	    </fo:block>
	    <fo:block
			font-size="10pt"
			space-before="0pt"
			space-after="0pt"
			>
		<xsl:text>Ave starts men: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='M']/@ave"/>
		<xsl:text> (</xsl:text><xsl:value-of select="$competitorStarts[@category='M']/@total"/>
		<xsl:text> starts, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='M']/@nStarters"/><xsl:text> competitors)</xsl:text>
	    </fo:block>
	    <fo:block
			font-size="10pt"
			space-before="0pt"
			space-after="0pt"
			>
		<xsl:text>Ave starts women: </xsl:text>
		<xsl:value-of select="$competitorStarts[@category='F']/@ave"/>
		<xsl:text> (</xsl:text><xsl:value-of select="$competitorStarts[@category='F']/@total"/>
		<xsl:text> starts, </xsl:text>
		<xsl:value-of select="$competitorSummary[@category='F']/@nStarters"/><xsl:text> competitors)</xsl:text>
	    </fo:block>
	</xsl:if>

	<!-- DIVISIONS CUP SUMMARY (special-case hack, oh well) -->
	<xsl:variable name="teamResults" 
		select="$standings/standingsSummary/teamSummary"/>
	<xsl:if test="$teamResults">
		<!-- assert( "$organizationType='ByAffiliation' and policy is Divisions Cup..." ) -->
		<!-- CONSIDER: maybe put this sucker in a table and be done with it?? -->
		<!-- FOP 0.20 doesn't support the table caption element -->
		<fo:block font-weight="bold" font-size="12pt" space-before="12pt" space-after="0pt">
			<xsl:text>Divisions Cup Standings (points per start)</xsl:text>
		</fo:block>
		<fo:table inline-progression-dimension="5.25in" table-layout="fixed" break-after="page"> 
			<!--class="summary"-->
			<fo:table-column column-width="0.5in"/>
			<fo:table-column column-width="1.5in"/>
			<fo:table-column column-width="0.75in"/>
			<fo:table-column column-width="1in"/>
			<fo:table-column column-width="0.75in"/>
			<fo:table-column column-width="0.75in"/>
			<fo:table-body>
			<fo:table-row font-size="10pt" font-weight="bold"> <!--class="summaryHeader"-->
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Place</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Team</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Pts/Strt</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt"># Competitors</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt"># Starts</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Points</fo:block></fo:table-cell>
			</fo:table-row>
			<xsl:for-each select="$teamResults">
				<fo:table-row font-size="10pt"> <!--class="summaryEntry"-->
				<fo:table-cell><fo:block>
				<xsl:choose>
					<xsl:when test="@status='QLF'">
						<xsl:value-of select="@posn"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:text>-</xsl:text>
					</xsl:otherwise>
				</xsl:choose>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="description"/>  <!-- @name has short name -->
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@ave"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@nRacers"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@nStarts"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@total"/>
				</fo:block></fo:table-cell>
				</fo:table-row>
			</xsl:for-each>
			</fo:table-body>
		</fo:table>
	</xsl:if>

	<!-- MEDAL COUNTS SUMMARY (special-case hack, oh well) -->
	<xsl:variable name="teamMedals" 
		select="$standings/standingsSummary/teamMedals"/>
	<xsl:if test="$teamMedals">
		<!-- assert( "$organizationType='ByAffiliation' and policy is Medal Counts..." ) -->
		<!-- CONSIDER: maybe put this sucker in a table and be done with it?? -->
		<!-- FOP 0.20 doesn't support the table caption element -->
		<fo:block font-weight="bold" font-size="12pt" space-before="12pt" space-after="0pt">
			<xsl:text>Medal Counts</xsl:text>
		</fo:block>
		<fo:table inline-progression-dimension="5.25in" table-layout="fixed" break-after="page"> 
			<!--class="summary"-->
			<fo:table-column column-width="0.5in"/>
			<fo:table-column column-width="1.5in"/>
			<fo:table-column column-width="0.75in"/>
			<fo:table-column column-width="0.5in"/>
			<fo:table-column column-width="0.5in"/>
			<fo:table-column column-width="0.75in"/>
			<fo:table-column column-width="1in"/>
			<fo:table-column column-width="0.75in"/>
			<fo:table-body>
			<fo:table-row font-size="10pt" font-weight="bold"> <!--class="summaryHeader"-->
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Place</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Team</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Medals</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Gold</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Silver</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Bronze</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt"># Competitors</fo:block></fo:table-cell>
				<fo:table-cell><fo:block space-before="6pt" space-after="3pt"># Medalists</fo:block></fo:table-cell>
			</fo:table-row>
			<xsl:for-each select="$teamMedals">
				<fo:table-row font-size="10pt"> <!--class="summaryEntry"-->
				<fo:table-cell><fo:block>
				<xsl:choose>
					<xsl:when test="@status='QLF'">
						<xsl:value-of select="@posn"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:text>-</xsl:text>
					</xsl:otherwise>
				</xsl:choose>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="description"/>  <!-- @name has short name -->
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@total"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@gold"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@silver"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@bronze"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@nRacers"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block>
				<xsl:value-of select="@nMedalists"/>
				</fo:block></fo:table-cell>
				</fo:table-row>
			</xsl:for-each>
			</fo:table-body>
		</fo:table>
	</xsl:if>

</xsl:template>


<!--
 ==============================================================================
 == Report footer section
 ==============================================================================
 -->

<xsl:template name="generateReportFooter">

	<xsl:variable name="classes">
		<xsl:call-template name="classesDescription">
			<xsl:with-param name="classes" select="event/classes"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:variable name="scoringElement">
		<xsl:call-template name="scoringElementDescription">
			<xsl:with-param name="scoringElementName" select="scoring/scoringElement"/>
		</xsl:call-template>
	</xsl:variable>

	<!-- div.id="reportFooter" -->

	<!-- class="subTitle" -->
    <fo:block
		font-size="12pt"
		font-weight="bold"
		space-before="12pt"
		space-after="0pt"
		>
		<xsl:text>Scoring Details</xsl:text>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:text>Classes: </xsl:text><xsl:value-of select="$classes"/>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:text>Scoring element: </xsl:text><xsl:value-of select="$scoringElement"/>
    </fo:block>
    <fo:block
		font-size="10pt"
		space-before="0pt"
		space-after="0pt"
		>
	<xsl:text>Scoring policy: </xsl:text><xsl:value-of select="scoring/scoringPolicy"/>
    </fo:block>

	<!-- class="subTitle" -->
    <fo:block
		font-size="12pt"
		font-weight="bold"
		space-before="12pt"
		space-after="0pt"
		>
		<xsl:text>Scored Races</xsl:text>
    </fo:block>
	<xsl:for-each select="$seriesRaces">
	    <fo:block
			font-size="10pt"
			space-before="0pt"
			space-after="0pt"
			>
		<xsl:value-of select="position()"/>
		<xsl:text>. </xsl:text>
		<xsl:call-template name="raceDescription">
			<xsl:with-param name="race" select="."/>
		</xsl:call-template>
		<xsl:text> - </xsl:text>
		<xsl:value-of select="courses"/>
	    </fo:block>
	</xsl:for-each>

	<xsl:variable name="multiCategoryRacers" 
		select="$standings/groupStandings/racerResult[@multipleResults='yes']"/>
			<!-- WAS: $allCompetitors[@nStarts] -->
	<xsl:if test="$multiCategoryRacers">
		<xsl:choose>
			<xsl:when test="$organizationType='ByClass'">
			<!-- class="subTitle" -->
		    <fo:block
				font-size="12pt"
				font-weight="bold"
				space-before="12pt"
				space-after="0pt"
				>
			<xsl:text>Racers with starts in both Open seed and age class</xsl:text>
		    </fo:block>
		    <fo:block
				font-size="10pt"
				space-before="0pt"
				space-after="0pt"
				>
				<xsl:text>Racer name - total number of starts (total number of finishes)</xsl:text>
		    </fo:block>
<!-- ####### TO DO [DJL 16-Nov-2002] ########### -->
		    <fo:block
				font-size="10pt"
				space-before="0pt"
				space-after="0pt"
				>
<xsl:text>***NOTE: Need to fix sorting!!</xsl:text>
		    </fo:block>
				<xsl:call-template name="generateMultiClassRacersReport">
					<xsl:with-param name="multiClassRacers" select="$multiCategoryRacers"/>
					<xsl:with-param name="index" select="1"/>
					<xsl:with-param name="prevAgeClass" select="'NONE'"/>
				</xsl:call-template>
			</xsl:when>
			<xsl:otherwise>
				<!-- //probably overzealous; if this is really serious, should detect long before now, eh??
				<xsl:message terminate="yes">***ERROR: should not have racers in multiple categories!!</xsl:message>
				-->
		    <fo:block
				font-size="10pt"
				space-before="12pt"
				space-after="0pt"
				font-weight="bold" font-style="italic"
				>
				<xsl:text>***WARNING: </xsl:text><xsl:value-of select="count($multiCategoryRacers)"/>
				<xsl:text> racers have starts in multiple categories - should not be!!!</xsl:text>
				</fo:block>
			</xsl:otherwise>
		</xsl:choose>
	</xsl:if>

</xsl:template>

<xsl:template name="generateMultiClassRacersReport">
<xsl:param name="multiClassRacers"/>
<xsl:param name="index"/>
<xsl:param name="prevAgeClass"/>

<!--########## TO DO [DJL 18-Jan-2003] ##################-->
		    <fo:block
				font-size="10pt"
				space-before="0pt"
				space-after="0pt"
				>
<xsl:text>###### TO DO: multi-class racers report goes here ########</xsl:text>
		    </fo:block>

</xsl:template>

<!--
 ==============================================================================
 == Report body
 ==============================================================================
 -->

<xsl:template name="generateReportBody">

	<xsl:if test="$DEBUG='yes'">
		<xsl:message>***PROCESSING &lt;standings&gt; organization <xsl:value-of select="$organizationType"/></xsl:message>
	</xsl:if>

	<!-- div.id="reportBody" -->
	<!-- FOP 0.20 doesn't support the table caption element -->
	<fo:block font-weight="bold" font-size="12pt" space-before="12pt" space-after="0pt">
		<xsl:value-of select="$standingsDescription"/>
	</fo:block>
	<fo:table inline-progression-dimension="7in" table-layout="fixed"> 
		<!--class="standings"-->
		<!-- FOP 0.20 requires column width specifications -->
		<fo:table-column column-width="0.4in"/>
		<fo:table-column column-width="2.5in"/>
		<fo:table-column column-width="0.5in"/>
		<fo:table-column column-width="0.3in"/>
		<fo:table-column column-width="0.3in"/>
		<xsl:for-each select="$seriesRaces">
			<fo:table-column column-width="0.75in"/>
		</xsl:for-each>
		<fo:table-body>
		<xsl:apply-templates select="$standings"/>
		</fo:table-body>
	</fo:table>

</xsl:template>

<xsl:template match="standings">

	<xsl:variable name="nGroups" select="count(groupStandings)"/>
	<xsl:variable name="includeCategoryHeaders" select="not (@singleCategory='yes')"/>  <!-- and nGroups > 1 -->

	<xsl:variable name="nColumns" select="5 + $nRaces"/>

	<xsl:if test="not($includeCategoryHeaders)">
		<fo:table-row font-size="10pt" font-weight="bold" keep-with-next="always"> <!--class="standingsHeader"-->
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Position</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Competitor</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">
				<xsl:call-template name="scoringElementLabel">
					<xsl:with-param name="scoringElementName" 
						select="$scoringRoot/scoring/scoringElement"/>
				</xsl:call-template>
			</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">St</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Fn</fo:block></fo:table-cell>
			<xsl:call-template name="generateRaceColumnLabels"/>
		</fo:table-row>
	</xsl:if>

	<xsl:for-each select="groupStandings">
		<xsl:call-template name="processResultGroup">
			<xsl:with-param name="includeCategoryHeaders" select="$includeCategoryHeaders"/>
			<xsl:with-param name="nColumns" select="$nColumns"/>
		</xsl:call-template>
	</xsl:for-each>

</xsl:template>

<xsl:template name="processResultGroup">
<xsl:param name="includeCategoryHeaders"/>
<xsl:param name="nColumns"/>

	<xsl:if test="$includeCategoryHeaders">
		<fo:table-row font-size="10pt" font-weight="bold" keep-with-next="always">  <!--class="groupHeader"-->
			<fo:table-cell number-columns-spanned="2"><fo:block space-before="6pt" space-after="3pt">
			<xsl:value-of select="description"/>
			</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">
				<xsl:call-template name="scoringElementLabel">
					<xsl:with-param name="scoringElementName" 
						select="$scoringRoot/scoring/scoringElement"/>
				</xsl:call-template>
			</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">St</fo:block></fo:table-cell>
			<fo:table-cell><fo:block space-before="6pt" space-after="3pt">Fn</fo:block></fo:table-cell>
			<xsl:call-template name="generateRaceColumnLabels"/>
		</fo:table-row>
	</xsl:if>

	<xsl:for-each select="racerResult">
		<xsl:call-template name="processRacerResult"/>
	</xsl:for-each>

	<!-- optional category summary -->
	<xsl:variable name="nQualifiers" select="@nQualifiers"/>
	<xsl:if test="$nQualifiers">
		<xsl:variable name="nCategoryCompetitors" select="count(racerResult)"/>
		<fo:table-row font-size="10pt">
		<xsl:call-template name="emptyTableCell"/>
		<fo:table-cell><fo:block font-style="italic"> <!--class="categorySummary"-->
			<xsl:text>(</xsl:text>
			<xsl:value-of select="$nQualifiers"/>
			<xsl:if test="$nQualifiers != $nCategoryCompetitors">
				<xsl:text> of </xsl:text>
				<xsl:value-of select="$nCategoryCompetitors"/>
			</xsl:if>
			<xsl:text> competitors</xsl:text>
			<xsl:if test="$nQualifiers != $nCategoryCompetitors">
				<xsl:text> qualified</xsl:text>
			</xsl:if>
			<xsl:text>)</xsl:text>
		</fo:block></fo:table-cell>
		<xsl:call-template name="emptyTableCell">
			<xsl:with-param name="count" select="$nColumns - 2"/>
		</xsl:call-template>
		</fo:table-row>
	</xsl:if>

</xsl:template>

<xsl:template name="processRacerResult">

	<xsl:variable name="racerID" select="@racerID"/>
	<xsl:variable name="racer" select="$allCompetitors[@id=$racerID]"/>

	<xsl:variable name="status" select="@status"/>
	<xsl:variable name="racerName">
		<xsl:call-template name="composeRacerNameEntry">
			<xsl:with-param name="racer" select="$racer"/>
			<xsl:with-param name="ageClass" select="@ageClass"/>  <!-- optional attr in OP/SS class standings -->
			<xsl:with-param name="includeRacerClass" 
				select="$includeRacerClass='yes'"/>
			<xsl:with-param name="includeRacerAffiliation" 
				select="$includeRacerAffiliation='yes'"/>
		</xsl:call-template>
	</xsl:variable>

	<xsl:if test="$status!='DNS'">
	<fo:table-row font-size="10pt">  <!--class="racerEntry"-->
		<xsl:choose>
			<xsl:when test="$status='QLF'">
				<fo:table-cell><fo:block> <!--class="position"-->
				<xsl:call-template name="positionString"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block> <!--class="name"-->
				<xsl:value-of select="$racerName"/>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block> <!--class="score"-->
					<xsl:call-template name="composeRacerScoreEntry">
						<xsl:with-param name="score" select="score"/>
						<xsl:with-param name="valueIfZero" select="'-'"/>
					</xsl:call-template>
				</fo:block></fo:table-cell>
			</xsl:when>
			<xsl:otherwise>
				<fo:table-cell><fo:block> <!--class="position"-->
				<xsl:text>-</xsl:text>
				</fo:block></fo:table-cell>
				<fo:table-cell><fo:block> <!--class="name"-->
				<xsl:value-of select="$racerName"/>
				</fo:block></fo:table-cell>
				<!-- ISSUE: report score if present, even though not qualified?? -->
				<xsl:call-template name="emptyTableCell"/>
			</xsl:otherwise>
		</xsl:choose>
		<fo:table-cell><fo:block> <!--class="sfCount"-->
			<xsl:call-template name="composeSeriesStartEntry">
				<xsl:with-param name="racerResult" select="."/>
				<xsl:with-param name="racer" select="$racer"/>
			</xsl:call-template>
		</fo:block></fo:table-cell>
		<fo:table-cell><fo:block> <!--class="sfCount"-->
			<xsl:call-template name="composeSeriesFinishEntry">
				<xsl:with-param name="racerResult" select="."/>
				<xsl:with-param name="racer" select="$racer"/>
			</xsl:call-template>
		</fo:block></fo:table-cell>
		<xsl:call-template name="generateRaceScoreEntries">
			<xsl:with-param name="racerResult" select="."/>
		</xsl:call-template>
	</fo:table-row>
	</xsl:if>

</xsl:template>

<xsl:template name="generateRaceColumnLabels">
	<xsl:variable name="useLineBreak" select="$nRaces &gt; 6"/>
	<xsl:for-each select="$seriesRaces">
		<fo:table-cell><fo:block space-before="6pt" space-after="3pt"> <!--class="rsScore"-->
			<xsl:value-of select="location/@name"/>
			<!-- HEURISTIC EXPERIMENT -->
<!-- ###########??? HOW TO DO LINE BREAK W/IN BLOCK?? #########
			<xsl:choose>
				<xsl:when test="$useLineBreak">
					<br/>
				</xsl:when>
				<xsl:otherwise>
					<xsl:text> </xsl:text>
				</xsl:otherwise>
			</xsl:choose>
##########################################-->
			<xsl:value-of select="discipline"/>
		</fo:block></fo:table-cell>
	</xsl:for-each>
</xsl:template>

<xsl:template name="generateRaceScoreEntries">
<xsl:param name="racerResult"/>
<xsl:param name="index" select="1"/>

	<xsl:variable name="raceScore" select="$racerResult/rsScore[@ri=$index]"/>

	<xsl:choose>
		<xsl:when test="$raceScore">
			<fo:table-cell><fo:block> <!--class="rsScore"-->
			<!-- #### TO DO (someday): if $racerResult/@counted='no' -> flag excluded score ####-->
			<xsl:value-of select="$raceScore"/>
			<xsl:if test="$raceScore/@posn">
				<xsl:text> (</xsl:text>
				<xsl:call-template name="seriesStartPositionString">
					<xsl:with-param name="posnAttr" select="$raceScore/@posn"/>
				</xsl:call-template>
				<xsl:text>)</xsl:text>
			</xsl:if>
			</fo:block></fo:table-cell>
		</xsl:when>
		<xsl:otherwise>
			<xsl:call-template name="emptyTableCell"/>
		</xsl:otherwise>
	</xsl:choose>

	<xsl:if test="not ($index = $nRaces)">
		<xsl:call-template name="generateRaceScoreEntries">
			<xsl:with-param name="racerResult" select="$racerResult"/>
			<xsl:with-param name="index" select="$index + 1"/>
		</xsl:call-template>
	</xsl:if>

</xsl:template>

</xsl:stylesheet>
